import logging
from django.conf import settings

from staff.lib.auth import auth_mechanisms as auth
from staff.lib.deploy_log_variables import DEPLOY_VARIABLES

try:
    import uwsgi
except ImportError:
    uwsgi = None


logger = logging.getLogger(__name__)


class UwsgiLogMiddleware(object):

    HOST_HEADERS = (
        'HTTP_X_FORWARDED_HOST_STAFF',
        'HTTP_X_FORWARDED_HOST',
        'HTTP_HOST',
    )

    def extract_actual_header(self, request, headers):
        for header_name in headers:
            ip = request.META.get(header_name)
            if ip:
                return ip
        return None

    if uwsgi:

        def process_request(self, request):
            self._set_deploy_vars()
            to_set = {
                'yandex_login': lambda: request.COOKIES.get('yandex_login'),
                'user_ip': lambda: self.extract_actual_header(request, settings.YAUTH_REAL_IP_HEADERS),
                'real_host': lambda: self.extract_actual_header(request, self.HOST_HEADERS),
                'request_id': lambda: request.context.context['request_id'],
            }
            for field, getter in to_set.items():
                try:
                    value = getter()
                    if value:
                        uwsgi.set_logvar(field, value)
                except Exception:
                    logger.exception('Exception on set uWSGI logvar {}'.format(field))

        def process_view(self, request, *args, **kwargs):
            try:
                uwsgi.set_logvar('auth_mechanism', str(request.auth_mechanism))
                uwsgi.set_logvar('view_name', request.resolver_match.view_name)
                if request.auth_mechanism == auth.OAUTH:
                    application = request.client_application
                    uwsgi.set_logvar('client_application', application.name and application.name.encode('utf-8'))
                    uwsgi.set_logvar('client_id', application.id)

                if not request.user.is_anonymous():
                    uwsgi.set_logvar('yandex_login', request.user.username)

                if request.auth_mechanism == 'tvm':
                    service_ticket = request.yauser.service_ticket
                    # user_ticket = request.yauser.user_ticket
                    uwsgi.set_logvar('tvm_client_id', str(service_ticket.src))

            except Exception:
                logger.exception('Exception on set uWSGI logvar')

        def _set_deploy_vars(self):
            for key in DEPLOY_VARIABLES:
                uwsgi.set_logvar(key, DEPLOY_VARIABLES[key])
