import contextlib
import functools

from django.db import models
from django.db.models.base import ModelBase
from django.utils.translation import ugettext_lazy as _
from django.utils.translation import get_language

from mptt.models import MPTTModel, MPTTModelBase
from mptt.managers import TreeManager

from staff.lib.db import atomic
from staff.lib.models.managers import FilterManager


class TimeStampedModel(models.Model):
    created_at = models.DateTimeField(
        auto_now_add=True, verbose_name=_('models.created_at-label')
    )

    modified_at = models.DateTimeField(
        auto_now=True, verbose_name=_('models.modified_at-label')
    )

    class Meta:
        abstract = True


class SoftDeletedModel(models.Model):
    is_active = models.BooleanField(
        default=True, verbose_name=_('models.is_active-label')
    )

    objects = models.Manager()
    active = FilterManager(is_active=True)
    inactive = FilterManager(is_active=False)

    class Meta:
        abstract = True


@contextlib.contextmanager
def context_suppress_autotime(model, fields):
    original_values = {}
    fields_to_change = [
        f for f in model._meta.local_fields
        if f.name in fields
    ]
    for field in fields_to_change:
        original_values[field.name] = {
            'auto_now': field.auto_now,
            'auto_now_add': field.auto_now_add,
        }
        field.auto_now = False
        field.auto_now_add = False
    try:
        yield
    finally:
        for field in fields_to_change:
            field.auto_now = original_values[field.name]['auto_now']
            field.auto_now_add = original_values[field.name]['auto_now_add']


def suppress_autotime(model, fields):
    def decorator(fu):
        @functools.wraps(fu)
        def decorated(*args, **kwargs):
            with context_suppress_autotime(model, fields):
                return fu(*args, **kwargs)
        return decorated
    return decorator


class IntranetManager(models.Manager):
    def active(self):
        return super(IntranetManager, self).filter(intranet_status__gt=0)

    def intranet_filter(self, *args, **kwargs):
        kwargs[str('intranet_status')] = 1
        return models.Manager.filter(self, *args, **kwargs)


class I18nProperty:
    def __init__(self, attr_name):
        self.attr_name = attr_name

    def __get__(self, instance, instance_type=None):

        native_lang = instance.native_lang

        if native_lang:
            lang = get_language()
            if lang:
                lang = lang.split('-')[0]

            # если выбранный язык не совпадает с native_lang, то
            # поле attr_name следует перевести на международный (английский).
            if lang != native_lang:
                translated = getattr(instance, self.attr_name + '_en')
                if translated:
                    return translated

        return getattr(instance, self.attr_name)


class I18nBaseModel(ModelBase):
    def __new__(mcs, *args, **kwargs):
        model = super(I18nBaseModel, mcs).__new__(mcs, *args, **kwargs)
        for field in model._meta.fields:
            field_parts = field.name.rsplit('_', 1)
            if field_parts[-1] == 'en':
                setattr(model, 'i_' + field_parts[0], I18nProperty(field_parts[0]))
        return model


class I18nMpttBaseModel(MPTTModelBase, I18nBaseModel):
    pass


def get_i_field(model_dict, field, prefix=''):
    en_field = prefix + field + '_en'
    nl_field = prefix + 'native_lang'
    if en_field in model_dict and nl_field in model_dict:
        lang = get_language()
        pos = lang.find('-')
        if pos > 0:
            lang = lang[:pos]
        if lang != model_dict[nl_field]:
            return model_dict[en_field]
    return model_dict[prefix+field]


class I18nModel(models.Model, metaclass=I18nBaseModel):
    """Основное правило: если у сущности есть интернационализованные поля
    (например office.name и office.name_en), то для правильной работы в поле
    native_lang должно быть записано значение языка, на котором записана
    информация в первом поле (для офиса office.name). Таким образом, если
    локаль человека не совпадает с native_lang для сущности, на которую он
    смотрит, то язык для него переключится на интернациональный английский.

    """
    native_lang = models.CharField(max_length=2, default='', blank=True)

    class Meta:
        abstract = True


class DisSoftDeletedModel(models.Model):
    intranet_status = models.IntegerField(default=1)

    objects = IntranetManager()
    all = models.Manager()

    def delete(self, using=None):
        raise Exception('Use intranet_status instead')

    class Meta:
        abstract = True


class AtomicSaveModel(models.Model):

    class Meta:
        abstract = True

    def save_base(self, raw=False, force_insert=False, force_update=False, using=None, update_fields=None):
        with atomic(using=using, savepoint=False):
            super().save_base(
                raw=raw,
                force_insert=force_insert,
                force_update=force_update,
                using=using,
                update_fields=update_fields,
            )


class IntranetModel(AtomicSaveModel, I18nModel, DisSoftDeletedModel):
    created_at = models.DateTimeField(editable=False)
    modified_at = models.DateTimeField(editable=False)

    class Meta:
        abstract = True
        app_label = 'django_intranet_stuff'


class TrackableModel(AtomicSaveModel, I18nModel, DisSoftDeletedModel):
    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True)
    version = models.IntegerField(default=0)

    def save_base(self, raw=False, force_insert=False, force_update=False, using=None, update_fields=None):
        self.version += 1
        super().save_base(raw, force_insert, force_update, using, update_fields)

    class Meta:
        abstract = True


class IntranetMpttManager(IntranetManager, TreeManager):
    pass


class I18nMpttModel(MPTTModel, I18nModel, metaclass=I18nMpttBaseModel):

    class Meta(MPTTModel.Meta):
        abstract = True
        app_label = 'django_intranet_stuff'

    tree = IntranetMpttManager()


class IntranetMpttModel(MPTTModel, IntranetModel, metaclass=I18nMpttBaseModel):

    class Meta(MPTTModel.Meta):
        abstract = True
        app_label = 'django_intranet_stuff'

    objects = IntranetMpttManager()
    tree = IntranetMpttManager()

    def _refresh_mptt_fields(self, instance: 'IntranetMpttModel'):
        opts = self._mptt_meta
        fields_to_refresh = (opts.left_attr, opts.right_attr, opts.level_attr, opts.tree_id_attr)
        instance.refresh_from_db(fields=fields_to_refresh)

    def save(self, *args, **kwargs):
        if self.parent and self.parent.id:
            self._refresh_mptt_fields(self.parent)

        if self.id and 'force_insert' not in kwargs:
            self._refresh_mptt_fields(self)

        super().save(*args, **kwargs)


class ColoredModel(models.Model):
    color = models.CharField(max_length=6, default='000000')  # RGB Black

    class Meta:
        abstract = True


class SortedModel(models.Model):
    position = models.IntegerField(default=0)

    class Meta:
        abstract = True
