from typing import Iterable, AnyStr, Dict, List, Any, Sequence, Set
from staff.departments.models import Department
from staff.lib.models.mptt import filter_by_heirarchy

DepChain = Dict[int, List[Dict[AnyStr, Any]]]

MAX_SIZE_FOR_SMALL_IMPL = 150


def _clean_func(value: Dict[str, Any], original_fields: Set[str]) -> Dict[str, Any]:
    return {
        k: v
        for k, v in value.items()
        if k in original_fields
    }


def get_departments_tree(departments: Sequence[int] = None, fields: Iterable[str] = None) -> DepChain:
    fields = set(fields or {'name', 'name_en', 'url', 'id'})
    departments = None if departments is None else set(departments)

    if departments is None:
        dep_qs = Department.all_types.values('id', 'parent_id', *fields).order_by('tree_id', 'lft')
        result = chains_dict_from_tree_list(dep_qs, lambda value: _clean_func(value, fields))
        return result

    if len(departments) == 0:
        return {}

    # По сформированму запросу получаем родителей для всех департаментов
    dep_qs = Department.all_types.values('id', 'parent_id', *fields)

    if len(departments) < MAX_SIZE_FOR_SMALL_IMPL:
        mptt_objects = Department.all_types.filter(id__in=departments).values('tree_id', 'lft', 'rght')
        dep_qs = filter_by_heirarchy(dep_qs, mptt_objects, by_children=False, include_self=True)

    dep_qs = dep_qs.order_by('tree_id', 'lft')

    result = chains_dict_from_tree_list(dep_qs, lambda value: _clean_func(value, fields))
    result = {
        key: value
        for key, value in result.items()
        if key in departments
    }
    return result


def chains_dict_from_tree_list(nodes, clean_func):
    result = {}
    for node in nodes:
        parent_chain = result[node['parent_id']] if node['parent_id'] else []
        chain = parent_chain[:]
        chain.append(clean_func(node))
        result[node['id']] = chain
    return result


class DepartmentsChain(object):
    """
    Класс для эффективного получения колбаски департаментов в виде списка с
    использованием кеширования. Флаг including_self определяет нужно ли
    включать сам департамент `department` в список
    """
    def __init__(self, including_self=True):
        self.departments_cache = {}
        self.including_self = including_self
        self._cache_hits = 0

    def get_chain_as_list(self, department=None, department_id=None):
        """
        Метод возвращает список родительских департаметов для экземпляра
        Department из аргумента `department` или id подразделения в
        `department_id`.
        """
        if department is None and department_id is None:
            raise Exception('No department given. I need department, bro.')

        if department:
            return self._get_chain_by_instance(department)
        elif department_id:
            return self._get_chain_by_id(department_id)

    def get_chain_as_str(self, department=None, department_id=None,
                         separator=' → '):
        """
        Метод возвращает список родительских департаметов в виде строки с
        разделителем `separator`
        """
        departments_chain = self.get_chain_as_list(department, department_id)
        return separator.join(d.i_name for d in departments_chain)

    def _get_chain_by_instance(self, department):
        """
        Метод возвращает список департаментов для экземпляра Department.
        """
        if department.id in self.departments_cache:
            self._cache_hits += 1
            return self.departments_cache[department.id]

        # if department.parent_id in self.departments_cache:
        #     self._cache_hits += 1
        #     chain = self.departments_cache[department.parent_id] + [department]
        #     self._put_in_cache(department.id, chain)
        #     return chain

        departments_chain = list(department.get_ancestors(ascending=False))
        if self.including_self:
            departments_chain.append(department)
        self._put_in_cache(department.id, departments_chain)
        return departments_chain

    def _get_chain_by_id(self, department_id):
        """
        Метод возвращает список департаментов для подразделения с
        идентификатором, `department_id`
        """
        if department_id in self.departments_cache:
            self._cache_hits += 1
            return self.departments_cache[department_id]

        try:
            department = Department.objects.get(pk=department_id)
        except Department.DoesNotExist:
            raise Exception('Department with id %d does not exist' % department_id)
        return self._get_chain_by_instance(department)

    def _put_in_cache(self, department_id, chain):
        """
        Метод кладет в кэш `value` по ключу `key` + проходит по текущей
        цепочке и добавляет в кэш дополнительные ключи
        """
        self.departments_cache[department_id] = chain

        parent_chain = chain[:-1]
        for level, department in enumerate(parent_chain, start=1):
            if department.id not in self.departments_cache:
                self.departments_cache[department.id] = parent_chain[:level]
