from typing import Union, Dict, TypeVar, Iterable

from mptt.models import MPTTModel

from django.db.models import Q, QuerySet


MpttModelObject = TypeVar('MpttModelObject', bound=MPTTModel)
MpttObject = Union[Dict, MpttModelObject]


def is_ancestor(parent: MpttObject, child: MpttObject):
    parent = get_mptt_keys(parent)
    child = get_mptt_keys(child)
    return (
        parent['lft'] <= child['lft']
        and
        parent['rght'] >= child['rght']
        and
        parent['tree_id'] == child['tree_id']
    )


def get_mptt_keys(mptt_object: MpttObject, prefix=''):
    if isinstance(mptt_object, dict):
        return {
            'tree_id': mptt_object[prefix + 'tree_id'],
            'lft': mptt_object[prefix + 'lft'],
            'rght': mptt_object[prefix + 'rght'],
        }
    else:
        return {
            'tree_id': mptt_object.tree_id,
            'lft': mptt_object.lft,
            'rght': mptt_object.rght,
        }


def get_heirarchy_filter_query(
    mptt_objects: Iterable[MpttObject],
    by_children: bool,
    filter_prefix: str = '',
    prefix: str = '',
    include_self: bool = False,
) -> Q:
    rght_operator = '__gte' if include_self else '__gt'
    lft_operator = '__lte' if include_self else '__lt'

    if by_children:
        rght_operator, lft_operator = lft_operator, rght_operator

    query = Q(pk=None)
    for mptt_object in mptt_objects:
        mptt_keys = get_mptt_keys(mptt_object, prefix)

        condition = {
            filter_prefix + 'tree_id': mptt_keys['tree_id'],
            filter_prefix + 'lft' + lft_operator: mptt_keys['lft'],
            filter_prefix + 'rght' + rght_operator: mptt_keys['rght'],
        }

        query |= Q(**condition)

    return query


def filter_by_heirarchy(
    query_set: QuerySet,
    mptt_objects: Iterable[MpttObject],
    by_children: bool,
    filter_prefix: str = '',
    prefix: str = '',
    include_self: bool = False,
) -> QuerySet:
    """
    :param query_set: QuerySet к которому надо добавить фильтрацию.
    :param mptt_objects: Список mptt объектов для фильтра.
    :param by_children: Брать детей, а иначе родителей.
    :param filter_prefix: Префикс поля для составления условия фильтра.
    :param prefix: Префикс поля, для получения атрибутов mptt. Нужно только, если список mptt - это словари.
    :param include_self: Включать или нет уровень самих фильтрующих объектов
    :return: Отфильтрованный QuerySet
    """
    query = get_heirarchy_filter_query(
        mptt_objects,
        by_children,
        filter_prefix,
        prefix,
        include_self
    )
    return query_set.filter(query)


def get_ancestors_query(mptt_objects: Union[MpttObject, Iterable[MpttObject]], include_self=False) -> Q:
    if not isinstance(mptt_objects, Iterable):
        mptt_objects = [mptt_objects]
    return get_heirarchy_filter_query(mptt_objects, by_children=False, include_self=include_self)


def get_descendants_query(mptt_objects: Union[MpttObject, Iterable[MpttObject]], include_self=False) -> Q:
    if not isinstance(mptt_objects, Iterable):
        mptt_objects = [mptt_objects]
    return get_heirarchy_filter_query(mptt_objects, by_children=True, include_self=include_self)
