import logging
from typing import Dict
from urllib3.util.url import url_attrs

from requests import sessions
from requests.models import parse_url
from requests.packages import urllib3
# for reexport
from requests import Response # noqa ignore=F401
from requests.exceptions import HTTPError, ConnectionError, ConnectTimeout, Timeout # noqa ignore=F401
from requests.exceptions import RequestException

from ids.registry import registry
from ids.repositories.base import RepositoryBase

from django.conf import settings

from staff.lib.log import log_context


urllib3.disable_warnings()


class HostDisabledException(Warning):
    pass


def check_disabled(host):
    if host in settings.DISABLED_HOSTS:
        raise HostDisabledException('request to {host} has been cancelled'.format(host=host))


def warn_on_cookies(logger, **request_kwargs: Dict[str, str]) -> None:
    headers = request_kwargs.get('headers', {})
    cookies = request_kwargs.get('cookies')
    header_names = {h.lower() for h in headers.keys()}
    if cookies or 'cookie' in header_names:
        logger.warning('Outgoing request with cookies!')


class Session(sessions.Session):
    logger_prefix = 'requests'

    @staticmethod
    def get_context(method, parsed_url, **kwargs):
        context = kwargs.copy()
        context['method'] = method.upper()
        for attr in url_attrs:
            context[attr] = getattr(parsed_url, attr)
        context.pop('files', None)
        context.pop('headers', None)
        context.pop('data', None)

        if kwargs.get('no_log_json', False):
            context.pop('json', None)

        return context

    def request(self, method, url, **kwargs):
        """
            log_message - текст, попадающий в лог.
            timeout - можно задать списком, тогда будут делаться ретраи.
            read_timeout - Если задан, то первый таймаут - на коннекшн, а этот - на чтение данных. Строго число.
        """

        parsed_url = parse_url(url)
        check_disabled(parsed_url.host)

        logger = logging.getLogger('{prefix}__{host}'.format(prefix=self.logger_prefix, host=parsed_url.host))

        timeouts = kwargs.pop('timeout', None)
        if not timeouts:
            logger.warning('No timeout set while requesting %s', url)
            timeouts = settings.REQUESTS_DEFAULT_TIMEOUTS.get(parsed_url.host, 2.0)

        read_timeout = kwargs.pop('read_timeout', None)

        if isinstance(timeouts, (int, float)):
            timeouts = (timeouts, )

        log_message = str(kwargs.pop('log_message', ''))
        no_log_json = kwargs.pop('no_log_json', False)

        for attempt_no, timeout in enumerate(timeouts, start=1):
            actual_timeout = timeout if not read_timeout else (timeout, read_timeout)
            try:
                response = super(Session, self).request(method, url, timeout=actual_timeout, **kwargs)
            except RequestException as e:
                if attempt_no >= len(timeouts):
                    logger.info('%s request to `%s` raises: %s', method, url, e, exc_info=True)
                    raise
            else:
                log_msg = ''
                if attempt_no > 1 or log_message:
                    log_msg = '(attempt: {no}; {msg})'.format(no=attempt_no, msg=log_message)

                elapsed = int(response.elapsed.total_seconds() * 1000)
                request_context = self.get_context(
                    method,
                    parsed_url,
                    elapsed=elapsed,
                    status_code=response.status_code,
                    size=len(response.content),
                    no_log_json=no_log_json,
                    **kwargs
                )
                with log_context(**request_context):
                    logger.info(
                        '%s>%s request to `%s` %s elapsed in %s ms.',
                        method.upper(), response.status_code, url, log_msg, elapsed,
                    )
                    warn_on_cookies(logger, **kwargs)
                return response


def get(url, params=None, **kwargs):
    session = Session()
    kwargs.setdefault('allow_redirects', True)
    return session.get(url, params=params, **kwargs)


def options(url, **kwargs):
    session = Session()
    kwargs.setdefault('allow_redirects', True)
    return session.options(url, **kwargs)


def head(url, **kwargs):
    session = Session()
    kwargs.setdefault('allow_redirects', False)
    return session.head(url, **kwargs)


def post(url, data=None, json=None, **kwargs):
    session = Session()
    return session.post(url, data=data, json=json, **kwargs)


def put(url, data=None, **kwargs):
    session = Session()
    return session.put(url, data=data, **kwargs)


def patch(url, data=None, **kwargs):
    session = Session()
    return session.patch(url, data=data, **kwargs)


def delete(url, **kwargs):
    session = Session()
    return session.delete(url, **kwargs)


def request(method, url, **kwargs):
    with Session() as session:
        return session.request(method=method, url=url, **kwargs)


def get_ids_repository(service: str, resource_type: str, **options) -> RepositoryBase:
    """
    Возвращает объект repository указанного ресурса с пропатченным session
    Вызывать вместо registry.get_repository()
    """

    repository = registry.get_repository(service=service, resource_type=resource_type, **options)
    session = Session()

    if hasattr(repository, 'connector'):
        repository.connector.session = session
    if hasattr(repository, 'client'):
        connection = getattr(repository.client, resource_type)._connection
        connection.session = session

    return repository
