import logging
from typing import List

from staff.lib.sync_tools.datagenerators import DataGenerator


class DataDiffMerger:
    """
    Сравнивает данные из базы и данные пришедшие из внешнего источника.
    Если нет данных из базы - объект создается.
    Если нет внешних данных - объект удаляется.
    Если есть и те и те данные - изменения накатываются на существующий объект.
    """

    def __init__(self, data_generator: DataGenerator, logger=None):
        self.created = 0
        self.updated = 0
        self.deleted = 0
        self.errors = 0
        self.skipped = 0
        self.all = 0

        self.data_generator = data_generator
        self.data_generator_name = self.data_generator.__class__.__name__

        if logger is None:
            logger = logging.getLogger(__name__)
        self.logger = logger

    def detect_action(self, ext_data, int_data):
        """
            Определяем действие, которое надо выполнить над объектом.
        """
        if int_data and ext_data:
            return 'update'

        if ext_data:
            return 'create'

        return 'delete'

    def on_update(self, obj):
        pass

    def on_updated(self, obj):
        pass

    def try_action(self, action, obj, sync_key):
        """
            Пытаемся выполнить действие над объектом.
            Пишем в лог. Считаем сколько каких действий выполнили.
        """
        msg = '%s %s %sd.'
        try:
            self.on_update(obj)
            if action in ('update', 'create'):
                obj.save()
            else:
                obj.delete()
            self.on_updated(obj)
        except Exception:
            self.logger.exception(msg % (action, self.data_generator_name, sync_key))
            self.errors += 1
        else:
            if action == 'create':
                self.created += 1
            if action == 'update':
                self.updated += 1
            if action == 'delete':
                self.deleted += 1

            self.logger.debug(msg % (self.data_generator_name, sync_key, action))

    def set_data(self, obj, data):
        for field, value in data.items():
            setattr(obj, field, value)

    def diff(self, ext_data, int_data):
        def diff_fields_gen(ext_data, int_data):
            for field in self.data_generator.get_diff_fields():
                if field in ext_data:
                    ext_value = ext_data.get(field)
                    int_value = int_data.get(field)
                    if (ext_value != int_value or
                            int_value is None and
                            ext_value == ''):
                        yield field, ext_value

        return dict(diff_fields_gen(ext_data, int_data))

    def _accepted_actions(self, create=False, update=False, delete=False) -> List[str]:
        result = []
        if create:
            result.append('create')
        if update:
            result.append('update')
        if delete:
            result.append('delete')

        return result

    def execute(self, create=False, update=False, delete=False):
        accepted_actions = self._accepted_actions(create, update, delete)

        for ext_data, int_data, sync_key in self.data_generator:
            self.logger.debug('Merging %s with %s by %s', ext_data, int_data, sync_key)
            self.all += 1

            action = self.detect_action(ext_data, int_data)
            self.logger.debug('Gonna %s', action)
            if action not in accepted_actions:
                self.logger.debug('%s is not accepted action, so skip', action)
                self.skipped += 1
                continue

            if action == 'delete':
                obj = self.data_generator.get_object(sync_key)
                self.logger.debug('Gonna %s %s', action, obj)
                if not obj:
                    self.skipped += 1
                    self.logger.error("Can't find object %s %s for delete.", self.data_generator_name, sync_key)
                    continue
                self.try_action(action, obj, sync_key)
                continue

            diff_data = self.diff(ext_data, int_data)

            if diff_data:
                self.logger.debug('Data differs by %s', diff_data)
                obj = self.data_generator.get_object(sync_key)
                if not obj:
                    self.skipped += 1
                    self.logger.warning("Can't find object %s %s.", self.data_generator_name, sync_key)
                    continue
                self.set_data(obj, diff_data)
                self.try_action(action, obj, sync_key)
            else:
                self.logger.debug('Object %s is up to date already', sync_key)
                self.skipped += 1

    @property
    def results(self):
        return vars(self)
