import attr
import logging

from ids.registry import registry
from ids.services.startrek2.repositories.base import create_repository
from startrek_client.objects import Resource

from django.conf import settings
from django.db import models

from staff.lib.log import log_context
from staff.lib.requests import get_ids_repository

registry.add_repository(*create_repository('translations'))
logger = logging.getLogger(__name__)


@attr.s(auto_attribs=True)
class FieldSyncParams(object):
    qs: models.QuerySet
    key_field: str
    translation_ru_field: str
    translation_en_field: str
    startrek_field: str
    key_prefix: str


class StartrekFieldsSync:
    def __init__(self, field_sync_params: FieldSyncParams) -> None:
        self._field_sync_params = field_sync_params
        auth_params = {
            'oauth_token': settings.ROBOT_STAFF_OAUTH_TOKEN,
            'user_agent': settings.STAFF_USER_AGENT,
        }
        # WARNING: Monkeypatching, ST wants PUT method instead of PATCH for translations
        self._translations = get_ids_repository(service='startrek2', resource_type='translations', **auth_params)
        self._translations._connection.patch = self._translations._connection.put
        self._fields = get_ids_repository(service='startrek2', resource_type='fields', **auth_params)
        self._model_title = self._field_sync_params.qs.model._meta.model_name.title()

    def _update_st_translations(self) -> None:
        logger.info(
            'Updating ST translations for %s model and ST field %s',
            self._model_title,
            self._field_sync_params.startrek_field,
        )

        for row in self._field_sync_params.qs:  # type: models.Model
            ru_text = getattr(row, self._field_sync_params.translation_ru_field)
            en_text = getattr(row, self._field_sync_params.translation_en_field)
            if not en_text:
                en_text = ru_text
            key = self._make_translation_key(row)

            if row.st_translation_id is not None:
                logger.info('Updating translation %s for row %s (key %s)', row.st_translation_id, row.pk, key)
                translation: Resource = self._translations.get_one({'id': row.st_translation_id})
                translation.update(value={'en': en_text, 'ru': ru_text})
            else:
                translations = [x for x in self._translations.get({}) if x.key == key]
                if not translations:
                    logger.info('Creating translation for row %s (key %s)', row.pk, key)
                    translation: Resource = self._translations.create(key=key, value={'en': en_text, 'ru': ru_text})
                    row.st_translation_id = translation.id
                else:
                    logger.info('%s translations found for %s', len(translations), row.pk)
                    row.st_translation_id = translations[0].id
                row.save()
                logger.info('Created translation for row %s, got translation id %s', row.pk, row.st_translation_id)

    def _get_key_field_value(self, row: models.Model) -> str:
        if '.' not in self._field_sync_params.key_field:
            key_field_value = getattr(row, self._field_sync_params.key_field, None)
            return key_field_value

        related_field_name, key_field_name = self._field_sync_params.key_field.split('.')
        related_object = getattr(row, related_field_name)
        key_field_value = getattr(related_object, key_field_name, None)
        return key_field_value

    def _make_translation_key(self, row: models.Model) -> str:
        key = f'{self._field_sync_params.key_prefix.lower()}{self._model_title}{self._get_key_field_value(row)}'
        replacements = (
            ('-', 'Minus'),
            ('&', 'And'),
            (' ', ''),
            ("'", ''),
            ('/', ''),
            ('_', ''),
        )
        for to_replace, replacement in replacements:
            key = key.replace(to_replace, replacement)

        return key

    def _update_st_field_choices(self) -> None:
        logger.info(
            'Updating ST %s field choices for %s model',
            self._field_sync_params.startrek_field,
            self._model_title,
        )
        translations_ids = list(
            self._field_sync_params.qs
            .filter(st_translation_id__isnull=False)
            .values_list('st_translation_id', flat=True)
        )

        translations = [
            {
                'id': translation_id,
                'self': f'{settings.STARTREK_API_URL}/v2/translations/{translation_id}',
            }
            for translation_id in translations_ids
        ]
        field = self._fields.get_one({'id': self._field_sync_params.startrek_field})
        field.update(optionsProvider={
            'type': 'TranslationOptionsProvider',
            'values': translations,
        })

    def sync(self) -> None:
        with log_context(model=self._model_title):
            self._update_st_translations()
            self._update_st_field_choices()
