import time
from typing import Dict, Any

import yenv
from celery import Task, exceptions

from staff.celery_app import app

from staff.lib.lock import lock_manager
from staff.lib.log import log_context
from staff.lib.startrek.issues import add_comment, create_issue, update_issue


LOCK_TIME = 10  # sec
IN_DEVELOPMENT = yenv.type == 'development'


def run_task(task, countdown=10, **kwargs):
    """Запускает выполнение указанного фонового задания.

    В зависимости от окружения выполняет отложено, либо в реальном времени.

    :param Callable task:
    :param int countdown: Отложить выполнение задания
        на указанное количество секунд (действует при отложенном выполнении).
    :param kwargs: Аргументы, которые будут переданы в функцию задания.

    """
    if IN_DEVELOPMENT:
        task(**kwargs)

    else:
        task.apply_async(kwargs=kwargs, countdown=countdown)


class LockedTask(Task):
    lock_name = None

    def run(self, *args, **kwargs):
        log_context_dict = self.get_log_context(*args, **kwargs)
        with log_context(
            task_name=self.name,
            task_id=self.request.id,
            expected_exceptions=[exceptions.Retry],
            **log_context_dict,
        ):
            nolock = kwargs.pop('nolock', False)
            if nolock or self.request.is_eager or yenv.type == 'development':
                return self.locked_run(*args, **kwargs)

            with lock_manager.lock(self.get_lock_name(*args, **kwargs), block=False) as is_locked:
                if is_locked:
                    start_time = time.time()
                    task_result = self.locked_run(*args, **kwargs)
                    task_duration = time.time() - start_time
                    if task_duration < LOCK_TIME:
                        time.sleep(LOCK_TIME - task_duration)
                    return task_result

    def get_log_context(self, *args, **kwargs):
        # type: (*Any, **Any) -> Dict
        return {}

    def get_lock_name(self, *args, **kwargs) -> str:
        name = self.lock_name
        if name:
            return name
        else:
            return self.__class__.__name__

    def locked_run(self, *args, **kwargs):
        raise self.ConfigurationError('Method locked_run must be overridden')

    class ConfigurationError(Exception):
        pass


@app.task
def create_issue_task(queue: str, summary: str, description: str, **fields) -> None:
    """
    Отложено создаёт тикет в Startrek.
    """
    create_issue(queue, summary, description, **fields)


@app.task
def update_issue_task(key: str, **fields) -> None:
    """
    Отложено редактирует тикет Startrek.
    """
    update_issue(key, **fields)


@app.task
def add_comment_task(key: str, text: str, **fields) -> None:
    """
    Отложено отправляет комментарий в тикет Startrek.
    """
    add_comment(key, text, **fields)
