from dateutil import parser
from datetime import date, timedelta
import pytz

from django.conf import settings
from django.forms import DateField


utc_tz = pytz.timezone('UTC')


def to_utc(datetime, local_tz=None, make_naive=True):
    # Дата она и в Африке дата
    if type(datetime) == date:
        return datetime
    if datetime.tzinfo is None:
        if not local_tz:
            local_tz = pytz.timezone(settings.TIME_ZONE)
        datetime = local_tz.localize(datetime)
    utc_dt = datetime.astimezone(utc_tz)
    if make_naive:
        utc_dt = utc_dt.replace(tzinfo=None)
    return utc_dt


def get_timedelta_setting(name, default=None):
    raw_value = getattr(settings, name, default)

    if isinstance(raw_value, timedelta):
        return raw_value

    elif isinstance(raw_value, int):
        return timedelta(seconds=raw_value)


def get_date_setting(name, default=None):
    raw_value = getattr(settings, name, default)

    if isinstance(raw_value, date):
        return raw_value

    elif isinstance(raw_value, str):
        return DateField().to_python(raw_value)


def parse_datetime(datetime_str):
    if datetime_str is None:
        return None

    if not isinstance(datetime_str, str):
        raise TypeError('datetime_str must be a string')

    try:
        value = parser.parse(datetime_str)
    except (AttributeError, ValueError):
        value = None
    return value


def parse_date(datetime_str):
    value = parse_datetime(datetime_str)
    if value:
        value = value.date()
    return value


class DateRange(object):
    def __init__(self, start, stop, step=None):
        """
        @type start: date
        @type stop: date
        @type step: timedelta | None
        """
        assert start < stop
        self.start = start
        self.stop = stop
        self.step = step or timedelta(1)

    def __iter__(self):
        current = self.start
        while current < self.stop:
            yield current
            current += self.step

    def __len__(self):
        return (self.stop - self.start).days // self.step.days
