import operator

import logging
logger = logging.getLogger(__name__)


class VoidType(dict):

    def __repr__(self):
        return 'Void'

    def __bool__(self):
        return False

    def __getitem__(self, key):
        return self


Void = VoidType()


def _getitem_or_void(obj, key):
    exception = None
    try:
        value = obj[key]
    except (IndexError, KeyError) as e:
        value = Void
        exception = e
    return value, exception


def diff_factory(left, right):
    return {
        (dict, dict): DictDiff,
        (VoidType, dict): DictDiff,
        (dict, VoidType): DictDiff,
        (list, list): ListDiff,
        # (set, set): SetDiff,
        (str, str): StringDiff,
        # (int, int): SimpleDiff,
        # (VoidType, int): SimpleDiff,
    }.get((type(left), type(right)), SimpleDiff)(left, right)


class Diff(object):

    def __init__(self, left, right):
        self.left = left
        self.right = right

    def __repr__(self):
        return ('{__classname__}({left}, {right})'
                .format(__classname__=self.__class__.__name__,
                        **vars(self)))

    @property
    def is_simple(self):
        return False

    @property
    def is_changed(self):
        return self.left != self.right

    @property
    def is_added(self):
        return self.is_changed and self.left == Void

    @property
    def is_removed(self):
        return self.is_changed and self.right == Void

    @property
    def pair(self):
        return self.left, self.right

    @property
    def changed_keys(self):
        return()

    @property
    def added_keys(self):
        return()

    @property
    def added_count(self):
        return len(self.added_keys)

    def as_dict_diff(self):
        return DictDiff(self.left, self.right)

    def as_list_diff(self):
        return ListDiff(self.left, self.right)

    # def as_set_diff(self):
    #     return SetDiff(self.left, self.right)

    def as_simple_diff(self):
        return SimpleDiff(self.left, self.right)


class SimpleDiff(Diff):
    @property
    def is_simple(self):
        return True

    def __getitem__(self, key):
        return None


class DictDiff(Diff):

    @property
    def all_keys(self):
        return set(self._left_keys) | set(self._right_keys)

    @property
    def common_keys(self):
        return set(self._left_keys) & set(self._right_keys)

    @property
    def removed_keys(self):
        return set(self._left_keys) - set(self._right_keys)

    @property
    def added_keys(self):
        return set(self._right_keys) - set(self._left_keys)

    @property
    def changed_keys(self):
        def changed(key):
            left_val, right_val = self._get(key)
            return operator.ne(left_val, right_val)

        return list(filter(changed, self.all_keys))

    @property
    def iterchanged(self):
        return self._iter_by(self.changed_keys)

    @property
    def iterall(self):
        return self._iter_by(self.all_keys)

    @property
    def itercommon(self):
        return self._iter_by(self.common_keys)

    def iteradded(self):
        return self._iter_by(self.added_keys)

    def added_items(self):
        list(self.iteradded())

    @property
    def iterremoved(self):
        return self._iter_by(self.removed_keys)

    def _items(self):
        for key in self.all_keys:
            left_val, right_val = self._get(key)
            yield key, diff_factory(left_val, right_val)

    def items(self):
        return list(self._items())

    @property
    def _left_keys(self):
        return list(self.left.keys())

    @property
    def _right_keys(self):
        return list(self.right.keys())

    def _iter_by(self, keys):
        for key in keys:
            left_val, right_val = self._get(key)
            yield diff_factory(left_val, right_val)

    def _get(self, key):
        left_val, right_val, _e = self._get_with_exception(key)
        return left_val, right_val

    def _get_with_exception(self, key):
        left_val, exception = _getitem_or_void(self.left, key)
        right_val, exception = _getitem_or_void(self.right, key)
        return left_val, right_val, exception

    def __iter__(self):
        return iter(self.all_keys)

    def __getitem__(self, key):
        left_val, right_val, exception = self._get_with_exception(key)

        if left_val == Void and right_val == Void:
            raise exception

        return diff_factory(left_val, right_val)


# В целях сравнения будем считать list частным случаем dict
class ListDiff(DictDiff):

    @property
    def iterchanged(self):
        for left_elem, right_elem in zip(self.left, self.right):
            if left_elem != right_elem:
                yield diff_factory(left_elem, right_elem)

    @property
    def _left_keys(self):
        return set(range(len(self.left)))

    @property
    def _right_keys(self):
        return set(range(len(self.right)))

    @property
    def added_list(self):
        return list(self.iteradded())

    @property
    def changed_list(self):
        return list(self.iterchanged)

    def __iter__(self):
        return self.iterall


class StringDiff(ListDiff):

    @property
    def iterchanged(self):
        for left_char, right_char in zip(self.left, self.right):
            if left_char != right_char:
                yield SimpleDiff(left_char, right_char)

    # def __iter__(self):
    #     return self.iterchanged

    # def __getitem__(self, name):
        # return diff_factory(self.left[name], self.right[name])


# class SetDiff(Diff):
#     pass


if __name__ == '__main__':
    dict1 = {
        'string': 'abc',
        'list': ['1', 2, {'key': 3}],
        'set': {1, '2'}
    }
    dict2 = {
        'string': 'abc',
        'list': ['1', 2, {'key': 3}, 4],
        'set': {1, '2'}
    }
    dict3 = {
        'string': 'abcd',
        'list': ['1', 2, {'key': 3}],
        'set': {1, '2', 3}
    }

    dd = DictDiff(dict1, dict2)
    assert not dd.is_changed
    print(dd.is_changed)

    dd = DictDiff(dict1, dict3)
    assert dd.is_changed
    print(dd['string'])
    print(dd['set'])
