from django.utils.translation import get_language

from staff.lib.utils import date


def localize(data_dict):
    is_en = get_language() != 'ru'
    fields = {f.rsplit('_', 1)[0]: f for f in data_dict if f.endswith('_en')}

    for field, en_field in fields.items():
        if is_en:
            data_dict[field] = data_dict[en_field]
        del data_dict[en_field]
    return data_dict


def split_localized_fields(data_dict):
    """
    Найти все локализуемые поля и переделать их в вид {'ru': ..., 'en': ...}
    """
    for key, value in data_dict.copy().items():
        if isinstance(value, dict):
            split_localized_fields(value)
        elif key.endswith('_en'):
            local_key = key[:-len('_en')]
            local_value = data_dict[local_key]
            data_dict[local_key] = {
                'ru': local_value,
                'en': value,
            }
            del data_dict[key]


def extract_related(data_dict, related_name, prefix=None, pop=True):
    related = {}

    for key in list(data_dict.keys()):
        if key.startswith(related_name):
            related_key = key.replace(related_name, '').lstrip('_')
            if prefix:
                related_key = '%s_%s' % (prefix, related_key)
            related[related_key] = data_dict.pop(key) if pop else data_dict.get(key)

    return related


def repack_related(data_dict):
    """
    Переделать `data_dict` вида
    {
        'name': 'Oxana'
        'office__name': 'Morozov',
        'office__city__name': 'Moscow'
    }
    в
    {
        'name': 'Oxana',
        'office': {
            'name': 'Morozov',
            'city': {
                'name': 'Moscow',
            }
        }
    }
    """
    repacked = {}
    for key, value in data_dict.items():
        if '__' in key:
            hierarchy = key.split('__')
            insert_hierarchically(repacked, hierarchy, value)
        else:
            repacked[key] = value
    return repacked


def insert_hierarchically(data_dict, hierarchy, value):
    """
    Положить value в data_dict по вложенному пути из списка hierarchy.
    """
    subdict = data_dict

    path, last_key = hierarchy[:-1], hierarchy[-1]
    for elem in path:
        subdict = subdict.setdefault(elem, {})
    subdict[last_key] = value


def normalize_serialized(data_dict, replacements=None, datetime_fields=()):
    replacements = replacements or {}

    data_dict = repack_related(data_dict)
    split_localized_fields(data_dict)

    for key_to_replace, replacement in replacements.items():
        if key_to_replace in data_dict:
            data_dict[replacement] = data_dict.pop(key_to_replace)

    for datetime_field in datetime_fields:
        dt_val = data_dict.get(datetime_field)
        if dt_val is None:
            continue

        if isinstance(dt_val, str):
            dt_val = date.parse_datetime(dt_val)

        dt_utc = date.to_utc(dt_val, make_naive=False)
        dt_utc_str = dt_utc.isoformat()
        data_dict[datetime_field] = dt_utc_str

    return data_dict
