import logging
from functools import wraps
from time import sleep
from typing import Tuple, Callable


def retry(
        delay: float,  # initial delay, in seconds (0.1 — 100 ms)
        backoff: float,  # multiplier to increase delay
        max_tries: int,  # limit of tries
        exceptions: Tuple[Exception] or Exception = Exception,  # exception class to catch, or a tuple
        logger: logging.Logger = None  # logger to write logs to
) -> Callable:
    def retry_decorator(f):
        assert delay > 0
        assert backoff >= 1
        assert max_tries > 0

        @wraps(f)
        def wrapper(*args, **kwargs):
            _tries = 0
            _delay = delay

            while True:
                try:
                    return f(*args, **kwargs)
                except exceptions:
                    _tries += 1
                    sleep(_delay)

                    _delay *= backoff
                    if logger:
                        logger.info(
                            "Couldn't get function %s results, tried %s out of %s attempts.",
                            f.__name__,
                            _tries,
                            max_tries,
                            exc_info=True,
                        )

                    if _tries >= max_tries:
                        raise

        return wrapper
    return retry_decorator
