from datetime import date
from typing import Iterable

import xlsxwriter

from django.http import HttpResponse

from staff.lib.xlsx.sheet_presenter import SheetPresenter
from staff.lib.xlsx.view import View
from staff.lib.xlsx.view_row import ViewRow, Column, ViewCell, DataValidation

from io import BytesIO


def make_xlsx_file_response(
    view_models: Iterable[SheetPresenter],
    file_name: str or None,
    file_name_prefix: str or None = None,
) -> HttpResponse:
    assert bool(file_name) ^ bool(file_name_prefix)

    stream = BytesIO()
    with xlsxwriter.Workbook(stream) as wb:
        for view_model in view_models:
            View(view_model).write_to_workbook(wb)

    data = stream.getvalue()

    response = HttpResponse(
        data,
        content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet; charset=utf-8',
        status=200,
    )

    file_name = file_name if file_name else '{}_{}'.format(file_name_prefix, date.today().isoformat())
    response['Content-Disposition'] = 'attachment; filename="%s.xlsx"' % file_name
    response['Content-Length'] = len(data)
    return response


__all__ = (
    'ViewRow',
    'Column',
    'ViewCell',
    'View',
    'SheetPresenter',
    'DataValidation',
    'make_xlsx_file_response',
)
