from staff.lib import requests
import logging

from django.conf import settings

from staff.celery_app import app
from staff.lib import tvm2
from staff.lib.db import atomic
from staff.lib.tasks import LockedTask

from .models import List

logger = logging.getLogger(__name__)
GET_MAILLISTS_TIMEOUTS = (10, 15, 30)


@app.task(ignore_result=True)
class UpdateMaillists(LockedTask):
    """Update maillists from ml.yandex-team.ru"""

    @atomic
    def locked_run(self):
        url = '{protocol}://{host}/apiv3/lists/info?include_unsub=1'.format(
            protocol=settings.ML_PROTOCOL,
            host=settings.ML_HOST,
        )
        tvm_ticket = tvm2.get_tvm_ticket_by_deploy('ml')
        if not tvm_ticket:
            logger.error('Error trying to retrieve tvm ticket. Task aborted.')
            return
        try:
            response = requests.get(
                url,
                headers={tvm2.TVM_SERVICE_TICKET_HEADER: tvm_ticket},
                timeout=GET_MAILLISTS_TIMEOUTS,
            )
        except requests.Timeout:
            logger.error(
                'Maillists request timeouted. url=%s, timeouts=%s',
                url,
                GET_MAILLISTS_TIMEOUTS,
            )
            return
        if response.status_code != 200:
            logger.error('Maillists responsed with code %s', response.status_code)
            return

        data = response.json()
        existing_maillists = {ml.name: ml for ml in List.objects.all()}

        if data and 'maillists' in data:
            for li in data['maillists']:
                lst = existing_maillists.pop(li['name'], None)
                if lst is not None:
                    is_new = True
                else:
                    lst = List(name=li['name'])
                    is_new = False
                if (
                    is_new
                    or lst.email != li['email']
                    or lst.info != li['info']
                    or lst.is_open != li['is_open']
                ):
                    lst.email = li['email']
                    lst.info = li['info']
                    lst.is_open = li['is_open']
                    lst.save()

            # Only deleted mail lists left in here. Kill them.
            for ml in existing_maillists.values():
                ml.delete()
