import pytz

from django.contrib import admin
from django.forms import ChoiceField

from staff.map.models import (
    City,
    Country,
    Device,
    Floor,
    FloorMap,
    Office,
    OfficePart,
    Placement,
    Room,
    Table,
    TableBook,
    TableReserve,
    Region,
)

from staff.lib.utils.admin import StaffModelAdmin, BaseIntranetAdmin, AutocompleteMixin
from staff.anketa.models import OfficeSetting


class TableBookAdmin(BaseIntranetAdmin):
    list_display = ('table', 'staff', 'date_from', 'date_to')
    ordering = ('-date_from', '-date_to')


class CountryAdmin(BaseIntranetAdmin):
    list_display = ('name', 'name_en', 'yandex_domain', 'code')
    search_fields = list_display


class CityAdmin(BaseIntranetAdmin):
    list_display = ('name', 'name_en')
    search_fields = list_display
    ordering = ('name', )


class OfficeAdmin(BaseIntranetAdmin):
    not_required_fields = (
        'address1',
        'address2',
        'address1_en',
        'address2_en',
        'fax',
        'phone'
    )
    list_display = (
        'name',
        'name_en',
        'is_virtual',
        'have_map',
        'address1',
        'address2',
        'address1_en',
        'address2_en'
    )
    search_fields = list_display
    ordering = ('name', )

    def get_form(self, *args, **kwargs):
        form = super(OfficeAdmin, self).get_form(*args, **kwargs)
        form.base_fields['tz'] = ChoiceField(
            choices=((zone, zone) for zone in pytz.common_timezones)
        )
        return form


class OfficeSettingAdmin(admin.ModelAdmin):
    list_display = (
        'office',
        'needs_food_money_info',
        'needs_software_info',
        'needs_hardware_info',
        'needs_vacation_info',
        'helpdesk_service',
        'cadre_service',
        'adaptation_service',
    )
    list_editable = list_display[1:]
    list_filter = ('office__city__country', 'office__city',)


class FloorAdmin(BaseIntranetAdmin):
    fields = (
        'intranet_status',
        'office',
        'native_lang',
        'name',
        'name_en',
        'num',
        'position',
        'img',
    )
    required_fields = [
        'native_lang',
        'name',
        'name_en',
        'num',
        'office',
        'position'
    ]

    list_display = 'office', 'name', 'name_en', 'num', 'position'
    list_filter = 'intranet_status',
    readonly_fields = 'img',
    search_fields = 'name', 'name_en', 'num', 'img',
    ordering = 'office',


class RoomAdmin(BaseIntranetAdmin):
    list_display = (
        'num',
        'name',
        'name_en',
        'name_exchange',
        'name_alternative',
        'room_type',
        'phone',
        'geometry',
    )
    list_display_links = ('num', 'name', 'name_en',)
    search_fields = list_display
    not_required_fields = (
        'name_exchange',
        'name_alternative',
        'color',
        'geometry',
    )


class DeviceAdmin(BaseIntranetAdmin):
    list_display = ('name', 'name_dns', 'description', 'floor')
    search_fields = list_display


class TableAdmin(BaseIntranetAdmin):
    list_display = ('num', )
    search_fields = list_display


class TableReserveAdmin(StaffModelAdmin):
    list_display = (
        'table',
        'department',
        'staff',
        'description',
        'occupied_at'
    )
    list_filter = ('department', )
    ordering = ('-occupied_at', )


class FloorMapAdmin(admin.ModelAdmin):
    list_display = (
        'file_name',
        'floor',
        'created_at',
        'is_ready',
    )
    list_filter = ('floor', )
    ordering = ('-created_at', )


class PlacementAdmin(admin.ModelAdmin):
    list_display = (
        'id',
        'name',
        'addr',
        'active_status',
        'organization',
        'office',
    )


class RegionAdmin(admin.ModelAdmin, AutocompleteMixin):
    group_field = 'group'
    list_display = (
        'id',
        'name',
        'floor',
        'group',
    )
    list_filter = ('floor__office', )
    ordering = ('floor__office', 'floor__num', )

    def formfield_for_foreignkey(self, db_field, request=None, **kwargs):
        autocomplete_field = self.maybe_autocomplete_formfield(db_field)
        if autocomplete_field is not None:
            return autocomplete_field

        return super(RegionAdmin, self).formfield_for_foreignkey(
            db_field, request, **kwargs)


class OfficePartAdmin(BaseIntranetAdmin):
    list_display = ('main_office', 'part_of_main_office')
    search_fields = list_display
    ordering = ('main_office', )


admin.site.register(City, CityAdmin)
admin.site.register(Country, CountryAdmin)
admin.site.register(Device, DeviceAdmin)
admin.site.register(Floor, FloorAdmin)
admin.site.register(FloorMap, FloorMapAdmin)
admin.site.register(Office, OfficeAdmin)
admin.site.register(OfficePart, OfficePartAdmin)
admin.site.register(OfficeSetting, OfficeSettingAdmin)
admin.site.register(Placement, PlacementAdmin)
admin.site.register(Room, RoomAdmin)
admin.site.register(Table, TableAdmin)
admin.site.register(TableBook, TableBookAdmin)
admin.site.register(TableReserve, TableReserveAdmin)
admin.site.register(Region, RegionAdmin)
