from datetime import timedelta
import sform

from django import forms
from django.core.exceptions import ValidationError

from staff.map.models import Floor, Office, Table, Room, Region, City

from staff.lib.validators import validate_login, validate_table_num
from staff.lib.validators import validate_room_num, validate_room_id, validate_region_id
from staff.lib.validators import validate_equipment_id, validate_table_id


class FloorFilter(forms.Form):
    floor_id = forms.ModelChoiceField(queryset=Floor.objects.active(), required=False, label='floor')


class RegionFilter(forms.Form):
    region_id = forms.ModelChoiceField(queryset=Region.objects.active(), required=False, label='region')


class OfficeFilter(forms.Form):
    office_id = forms.ModelChoiceField(queryset=Office.objects.active(), required=False, label='office')


class TableFilter(forms.Form):
    floor_id = forms.ModelChoiceField(queryset=Floor.objects.active(),
                                      required=False)
    x = forms.IntegerField(required=False)
    y = forms.IntegerField(required=False)
    distance = forms.IntegerField(required=False)
    table_num = forms.IntegerField(required=False,
                                   validators=[validate_table_num])
    table_id = forms.IntegerField(required=False,
                                  validators=[validate_table_id])
    login = forms.CharField(required=False, validators=[validate_login])

    def clean(self):
        """
        Если приходит table_num или login, то ищем стол и устанавливаем
        x, y, floor_id
        """
        cleaned_data = super(TableFilter, self).clean()
        table_num = cleaned_data.get('table_num')
        table_id = cleaned_data.get('table_id')
        login = cleaned_data.get('login')
        distance = cleaned_data.get('distance')

        table = None
        if table_id:
            table = Table.objects.get(id=table_id)
        elif table_num:
            table = Table.objects.get(id=table_num)
        elif login:
            try:
                table = Table.objects.get(staff__login=login)
            except Table.DoesNotExist:
                pass

        if table:
            cleaned_data['floor_id'] = table.floor_id
            cleaned_data['x'] = table.coord_x
            cleaned_data['y'] = table.coord_y

        if distance is None:
            cleaned_data['distance'] = 100

        return cleaned_data

    def has_x_and_y(self):
        return self.cleaned_data['x'] is not None and self.cleaned_data['y'] is not None


class RoomFinder(forms.Form):
    room_num = forms.IntegerField(required=False, validators=[validate_room_num])
    name_exchange = forms.CharField(required=False)
    room_id = forms.IntegerField(required=False, validators=[validate_room_id])

    def clean(self):
        cleaned_data = super(RoomFinder, self).clean()
        room_num = cleaned_data.get('room_num')
        name_exchange = cleaned_data.get('name_exchange')
        room_id = cleaned_data.get('room_id')

        if not any((room_num, name_exchange, room_id)):
            raise ValidationError('room_num, room_id or name_exchange must be in params')
        elif name_exchange and not room_id:
            try:
                room = Room.objects.get(name_exchange=name_exchange, intranet_status=1)
                cleaned_data['room_id'] = room.id
            except Room.DoesNotExist:
                raise ValidationError('Нет комнаты %s' % name_exchange)
        elif room_num and not room_id:
            try:
                room = Room.objects.get(num=room_num, intranet_status=1)
                cleaned_data['room_id'] = room.id
            except Room.DoesNotExist:
                raise ValidationError('Нет комнаты номер %s' % room_num)
        return cleaned_data


class RoomUsageFilter(forms.Form):
    floor_id = forms.ModelChoiceField(queryset=Floor.objects.active(), required=True, label='floor')
    date_from = forms.DateField(required=True)
    date_to = forms.DateField(required=True)

    def clean(self):
        cleaned_data = super().clean()

        if not self.is_valid():
            return

        if cleaned_data['date_from'] > cleaned_data['date_to']:
            self._errors['date_from'] = ['date_from should be before date_to']

        return cleaned_data


class RegionFinder(forms.Form):
    region_id = forms.IntegerField(required=False, validators=[validate_region_id])

    def clean(self):
        cleaned_data = super(RegionFinder, self).clean()
        region_id = cleaned_data.get('region_id')

        if not region_id:
            raise ValidationError('region_id must be in params')

        return cleaned_data


class EquipmentFinder(forms.Form):
    equipment_id = forms.IntegerField(
        required=True,
        validators=[validate_equipment_id],
    )


class EditHistoryForm(forms.Form):
    date_from = forms.DateField(required=True)
    date_to = forms.DateField(required=True)

    def clean(self):
        cleaned_data = super(EditHistoryForm, self).clean()

        if not self.is_valid():
            return cleaned_data

        if cleaned_data['date_from'] > cleaned_data['date_to']:
            raise ValidationError('date_from should be before date_to')

        cleaned_data['date_to'] += timedelta(days=1)

        return cleaned_data


class ExportRoomsUsageForm(sform.SForm):
    date_from = sform.DateField(state=sform.REQUIRED)
    date_to = sform.DateField(state=sform.REQUIRED)
    office = sform.ModelChoiceField(
        queryset=Office.objects.filter(intranet_status=1),
        label_extractor='name',
    )
    floor = sform.ModelChoiceField(
        queryset=Floor.objects.filter(intranet_status=1),
        label_extractor='name',
    )
    city = sform.ModelChoiceField(
        queryset=City.objects.filter(intranet_status=1),
        label_extractor='id',
    )
