from datetime import datetime, date
from typing import Any, Dict, List, Iterable

from staff.person.models import Staff
from staff.lib.models.base import get_i_field
from staff.lib import xlsx

from staff.map.models import TableReserve, TableBook
from staff.map.utils import get_map_room_id_to_room_square


def _convert_obj(obj: dict, name_map: dict) -> dict:
    result = {}
    for k, v in obj.items():
        key = name_map.get(k, k)

        value = v
        if isinstance(v, (datetime, date)):
            value = value.strftime('%d.%m.%Y')
        result[key] = value
    return result


class TableSet:
    """Контроллер для множества столов"""
    def __init__(self, tables):
        self.tables = tables

    def _build_dict(self):
        self.tables_dict = {table['id']: table for table in self.tables}

    def _prefetch_related(self):
        self.employees = (
            Staff.objects
            .filter(table_id__in=list(self.tables_dict.keys()))
            .filter(is_dismissed=False)
            .values(
                'table_id',
                'login',
                'first_name',
                'last_name',
                'position',
                'department__bg_color',
            )
        )

        self.reservations = (
            TableReserve.objects
            .active()
            .filter(table_id__in=self.tables_dict.keys())
            .values(
                'table_id',
                'description',
                'occupied_at',
                'staff__login',
                'staff__first_name',
                'staff__last_name',
                'department_id',
                'department__bg_color',
                'department__url',
                'department__native_lang',
                'department__name',
                'department__name_en',
            )
        )

        cur_date = date.today()
        self.booked = (
            TableBook.objects
            .filter(table_id__in=self.tables_dict.keys())
            .filter(date_to__gte=cur_date)
            .filter(date_from__lte=cur_date)
            .values(
                'table_id',
                'description',
                'date_from',
                'date_to',
                'staff__login',
                'staff__first_name',
                'staff__last_name',
                'staff__department_id',
                'staff__department__bg_color',
                'staff__department__url',
                'staff__department__native_lang',
                'staff__department__name',
                'staff__department__name_en',
            )
        )

    def _bind_empty_lists(self):
        for table in self.tables_dict.values():
            table['employees'] = []
            table['reservations'] = []
            table['booked'] = []

    def _bind_employees(self):
        for employee in self.employees:
            table_id = employee.pop('table_id')
            emps_at_table = self.tables_dict[table_id]['employees']

            name_map = {'department__bg_color': 'department_color'}

            emps_at_table.append(_convert_obj(employee, name_map))

    def _bind_reservations(self):
        for res in self.reservations:
            table_id = res.pop('table_id')
            reserves_on_table = self.tables_dict[table_id]['reservations']

            department_url = res.pop('department__url')
            if department_url:
                department_url = '/departments/' + department_url

            department_name = get_i_field(res, 'name', 'department__')
            del res['department__native_lang']
            del res['department__name']
            del res['department__name_en']

            name_map = {
                'staff__login': 'login',
                'staff__first_name': 'first_name',
                'staff__last_name': 'last_name',
                'department__bg_color': 'department_color',
                }

            converted_res = _convert_obj(res, name_map)
            converted_res['department_name'] = department_name
            converted_res['department_url'] = department_url

            reserves_on_table.append(converted_res)

    def _bind_booked(self):
        for book in self.booked:
            table_id = book.pop('table_id')
            booked_on_table = self.tables_dict[table_id]['booked']

            department_name = get_i_field(book, 'name', 'staff__department__')
            del book['staff__department__native_lang']
            del book['staff__department__name']
            del book['staff__department__name_en']

            name_map = {
                'staff__login': 'login',
                'staff__first_name': 'first_name',
                'staff__last_name': 'last_name',
                'staff__department__bg_color': 'department_color',
                'staff__department_id': 'department_id'
                }

            converted_book = _convert_obj(book, name_map)
            converted_book['department_name'] = department_name

            booked_on_table.append(converted_book)

    def bind_additional_info(self):
        self._build_dict()
        self._prefetch_related()
        self._bind_empty_lists()
        self._bind_employees()
        self._bind_reservations()
        self._bind_booked()

    def get_objects(self):
        return list(self.tables_dict.values())


class RoomSet:
    """Контроллер для множества комнат"""
    __room_squares_map = None

    def __init__(self, rooms: List[Dict[str, Any]]):
        self.rooms = rooms

    def _build_dict(self):
        self.rooms_dict: Dict[int, Dict[str, Any]] = {room['id']: room for room in self.rooms}

    def _prefetch_related(self):
        self.employees = (
            Staff.objects
            .filter(room_id__in=list(self.rooms_dict.keys()), table_id=None, is_dismissed=False)
            .values(
                'room_id',
                'login',
                'first_name',
                'last_name',
                'position',
                'department__bg_color',
            )
        )

    def _bind_empty_lists(self):
        for room in self.rooms_dict.values():
            room['employees'] = []

    def _bind_employees(self):
        for employee in self.employees:
            room_id = employee.pop('room_id')
            emps_at_room = self.rooms_dict[room_id]['employees']

            name_map = {'department__bg_color': 'department_color'}

            emps_at_room.append(_convert_obj(employee, name_map))

    def _bind_squares(self):
        if self.__room_squares_map is None:
            self.__room_squares_map = get_map_room_id_to_room_square()

        for room in self.rooms_dict.values():
            room['square'] = self.__room_squares_map.get(room['id'])

    def bind_additional_info(self):
        self._build_dict()
        self._prefetch_related()
        self._bind_empty_lists()
        self._bind_employees()
        self._bind_squares()

    def get_objects(self):
        return list(self.rooms_dict.values())


_HEADINGS = [
    'Комната ID',
    'Номер комнаты',
    'Этаж ID',
    'Комната название',
    'Офис ID',
    'Офис название',
    'Средняя загруженность по количеству мест',
    'Минимальная загруженность по количеству мест',
    'Максимальная загруженность по количеству мест',
    'Медианная загруженность по количеству мест',
    'Количество столов в комнате',
    'Средняя загруженность по площади',
    'Минимальная загруженность по площади',
    'Максимальная загруженность по площади',
    'Медианная загруженность по площади',
]


class RoomsUsagePresenter(xlsx.SheetPresenter):
    sheet_name = 'rooms_usage'

    _font_base = {
        'font_size': 12,
    }

    _heading_row_format = {
        'align': 'center',
        'valign': 'vcenter',
        'text_wrap': True,
        'bold': True,
        'bg_color': '#B4C6E7',
        **_font_base,
    }

    _data_row_format = {
        'border': 1,
        **_font_base,
    }
    fields_for_xlsx = (
        'id',
        'num',
        'floor_id',
        'name',
        'floor__office_id',
        'floor__office__name',
        'avg_usage',
        'min_usage',
        'max_usage',
        'median_usage',
        'table_count',
        'avg_square_based_usage',
        'min_square_based_usage',
        'max_square_based_usage',
        'median_square_based_usage',
    )

    def __init__(self, model: Iterable):
        super().__init__(model=model)

    def columns(self) -> Iterable[xlsx.Column]:
        return self.auto_columns()

    def rows(self) -> Iterable[xlsx.ViewRow]:
        yield self.column_captions_row()
        yield from self.data_rows()

    def data_rows(self):
        for row in self._model:
            yield self.default_view_row_from_row_data(row[field] for field in self.fields_for_xlsx)

    def column_captions_row(self) -> xlsx.ViewRow:
        return xlsx.ViewRow(
            cells=xlsx.SheetPresenter.default_view_cells_from_row_data(_HEADINGS),
            row_format=self._heading_row_format,
        )
