import csv
import datetime
import logging
from typing import Any, Dict, List, Set

from django.core.exceptions import PermissionDenied
from django.conf import settings
from django.contrib.auth.decorators import permission_required
from django.db import IntegrityError
from django.db.models import Avg, Min, Max, Count
from django.db.models.aggregates import Aggregate
from django.http import JsonResponse, HttpRequest
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_POST, require_http_methods, require_GET

from staff.gap.api.forms import WEEKDAYS
from staff.lib.decorators import responding_json, available_for_external, make_json_response
from staff.lib.forms import errors as form_errors
from staff.lib.utils.qs_values import extract_related, localize
from staff.lib.xlsx import make_xlsx_file_response
from staff.preprofile.forms import FileImportForm
from staff.person.models import Staff
from staff.workspace_management import workspace_management_api

from staff.map.api.forms import (
    FloorFilter,
    OfficeFilter,
    TableFilter,
    RoomFinder,
    RoomUsageFilter,
    RegionFinder,
    EditHistoryForm,
    EquipmentFinder,
    ExportRoomsUsageForm,
)
from staff.map.api.objects import TableSet, RoomSet, RoomsUsagePresenter
from staff.map.api.rooms_export import RoomsExportPresenter, rooms_export_model
from staff.map.forms.region import RegionForm
from staff.map.models.logs import Log
from staff.map.models import (
    FloorMap,
    Device,
    Office,
    Floor,
    Room,
    Region,
    Table,
    ROOM_TYPES,
    SECURITY_DEVICES,
    SourceTypes,
    StaffOfficeLog,
)
from staff.map.utils import get_map_room_id_to_room_square

logger = logging.getLogger(__name__)


def _add_share_pies_to_rooms_qs_result(result: List[Dict[str, Any]]) -> None:
    share_pies = workspace_management_api.get_actual_share_pies_for_rooms([row['id'] for row in result])
    for row in result:
        row['share_pie'] = {}
        if row['id'] in share_pies:
            share_pie = share_pies[row['id']]
            row['share_pie']['room_area'] = share_pie.room_area
            row['share_pie']['shares'] = [
                {'business_unit_id': share.business_unit_id, 'share_value': share.share_value}
                for share in share_pie.shares
            ]


@available_for_external('map.external_with_map_access')
@responding_json
def offices(request):
    offices_list = (
        Office.objects
        .active()
        .order_by('city__country__position', 'city__position', 'position')
        .values(
            'id',
            'name',
            'name_en',
            'position',
            'code',
            'have_map',
            'city__id',
            'city__position',
            'city__name',
            'city__name_en',
            'city__country__id',
            'city__country__position',
            'city__country__name',
            'city__country__name_en',
        )
    )

    result = []

    for office in offices_list:
        localize(office)
        office['country'] = extract_related(office, 'city__country')
        office['city'] = extract_related(office, 'city')
        result.append(office)

    return result


FLOOR_FIELDS = (
    'id',
    'name',
    'num',
    'office_id',
    'position',
    'img',
)


@available_for_external('map.external_with_map_access')
@responding_json
def floors(request):
    objects = (
        Floor.objects
        .filter(intranet_status=1, office__intranet_status=1)
        .order_by('position')
    )

    form = OfficeFilter(request.GET or None)
    if form.is_valid() and form.cleaned_data['office_id']:
        objects = objects.filter(office_id=form.cleaned_data['office_id'])
    elif form.errors:
        return {'errors': form.errors}

    floors_list = []
    for floor_dict in objects.values(*FLOOR_FIELDS):
        floor_dict = add_floor_map_fields(floor_dict)
        if floor_dict:
            floors_list.append(floor_dict)

    return floors_list


def add_floor_map_fields(floor_dict):
    try:
        floor_map = (
            FloorMap.objects
            .filter(floor_id=floor_dict['id'], is_ready=True)
            .values('file_name', 'max_zoom', 'min_zoom')
            .order_by('-created_at')
        )[0]
    except IndexError:
        return None

    floor_dict.update(floor_map)
    return floor_dict


TABLE_FIELDS = (
    'id',
    'num',
    'floor_id',
    'coord_x',
    'coord_y',
    'floor__office__code',
)


@available_for_external('map.external_with_map_access')
@responding_json
def table(request):
    objects = Table.objects.active()

    form = TableFilter(request.GET or None)
    if form.is_valid():
        floor_id = form.cleaned_data.get('floor_id')
        x = form.cleaned_data.get('x')
        y = form.cleaned_data.get('y')
        distance = form.cleaned_data.get('distance')

        if floor_id:
            objects = objects.filter(floor_id=floor_id)
            if form.has_x_and_y():
                objects = objects.filter(
                    coord_x__gte=x - distance,
                    coord_x__lte=x + distance,
                    coord_y__gte=y - distance,
                    coord_y__lte=y + distance,
                )

    elif form.errors:
        return {'errors': form.errors}, 404

    tables_ = TableSet(tables=list(objects.values(*TABLE_FIELDS)))
    tables_.bind_additional_info()

    return tables_.get_objects()


@available_for_external('map.external_with_map_access')
@responding_json
def tables(request):
    objects = Table.objects.active()

    form = FloorFilter(request.GET or None)
    if form.is_valid() and form.cleaned_data['floor_id']:
        objects = objects.filter(floor_id=form.cleaned_data['floor_id'])
    elif form.errors:
        return {'errors': form.errors}

    tables_ = TableSet(tables=list(objects.values(*TABLE_FIELDS)))
    tables_.bind_additional_info()

    #    return HttpResponse('<body></body>')
    return tables_.get_objects()


ROOM_FIELDS = (
    'id',
    'num',
    'floor_id',
    'name',
    'name_alternative',
    'room_type',
    'geometry',
    'coord_x',
    'coord_y',
    'floor__office__code',
    'additional'
)


@available_for_external('map.external_with_map_access')
@responding_json
def room(request):
    form = RoomFinder(request.GET)
    if form.is_valid():
        objects = Room.objects.filter(id=form.cleaned_data['room_id'])
        room = objects[0]
    else:
        return {'errors': form.errors}, 404

    if room.room_type == ROOM_TYPES.CONFERENCE:
        return {'errors': {'room_type': ['wrong room type']}}, 404

    rooms_ = RoomSet(rooms=[objects.values(*ROOM_FIELDS)[0]])
    rooms_.bind_additional_info()
    result = rooms_.get_objects()

    if workspace_management_api.user_can_manage_workspace_areas(request.user):
        _add_share_pies_to_rooms_qs_result(result)

    return result[0]


@available_for_external('map.external_with_map_access')
@responding_json
def rooms(request):
    BASIC_ROOM_TYPES = (
        ROOM_TYPES.OFFICE,
        ROOM_TYPES.ROOM_WAREHOUSE,
        ROOM_TYPES.ROOM_SERVICE,
        ROOM_TYPES.ROOM_STAIRS,
        ROOM_TYPES.ROOM_SWITCH,
        ROOM_TYPES.ROOM_HALLWAY,
    )
    objects = Room.objects.active().filter(room_type__in=BASIC_ROOM_TYPES)

    form = FloorFilter(request.GET or None)
    if form.is_valid() and form.cleaned_data['floor_id']:
        objects = objects.filter(floor_id=form.cleaned_data['floor_id'])
    elif form.errors:
        return {'errors': form.errors}

    rooms_ = RoomSet(rooms=list(objects.values(*ROOM_FIELDS)))
    rooms_.bind_additional_info()
    result = rooms_.get_objects()

    if workspace_management_api.user_can_manage_workspace_areas(request.user):
        _add_share_pies_to_rooms_qs_result(result)

    return result


def annotate_rooms_usage(qs):
    return (
        qs
        .annotate(avg_usage=Avg('roomusage__usage'))
        .annotate(min_usage=Min('roomusage__usage'))
        .annotate(max_usage=Max('roomusage__usage'))
        .annotate(
            median_usage=Aggregate(
                'roomusage__usage',
                function='percentile_cont',
                template='%(function)s(0.5) WITHIN GROUP (ORDER BY %(expressions)s)',
            )
        )
        .annotate(avg_square_based_usage=Avg('roomusage__square_based_usage'))
        .annotate(min_square_based_usage=Min('roomusage__square_based_usage'))
        .annotate(max_square_based_usage=Max('roomusage__square_based_usage'))
        .annotate(
            median_square_based_usage=Aggregate(
                'roomusage__square_based_usage',
                function='percentile_cont',
                template='%(function)s(0.5) WITHIN GROUP (ORDER BY %(expressions)s)',
            )
        )
        .annotate(table_count=Count('table', distinct=True))
    )


@available_for_external('map.external_with_map_access')
@responding_json
def rooms_usage(request):
    form = RoomUsageFilter(request.GET or None)
    if not form.is_valid():
        return JsonResponse(data=form.errors, status=400)

    qs = (
        Room.objects
        .filter(
            room_type=ROOM_TYPES.OFFICE,
            intranet_status=1,
            floor_id=form.cleaned_data['floor_id'],
            roomusage__date__gte=form.cleaned_data['date_from'],
            roomusage__date__lte=form.cleaned_data['date_to'],
        )
        .values(*ROOM_FIELDS)
    )

    rooms = annotate_rooms_usage(qs)

    room_squares = get_map_room_id_to_room_square()
    for room in rooms:
        room['square'] = room_squares.get(room['id'])

    return {
        'rooms_usage': list(rooms),
        'limit_usage': settings.MAP_LIMIT_ROOM_USAGE,
    }


ROOM_EXPORT_FIELDS = (
    'id',
    'num',
    'floor_id',
    'name',
    'floor__office_id',
    'floor__office__name',
)


@require_http_methods(['GET'])
def export_rooms_usage(request: HttpRequest):
    form = ExportRoomsUsageForm(data=request.GET)
    file_name = 'rooms_usage'

    if not form.is_valid():
        return make_json_response(request, form.errors, status=404)

    qs = (
        Room.objects
        .filter(
            room_type=ROOM_TYPES.OFFICE,
            intranet_status=1,
            roomusage__date__gte=form.cleaned_data['date_from'],
            roomusage__date__lte=form.cleaned_data['date_to'],
        )
    )
    if form.cleaned_data['office']:
        qs = qs.filter(floor__office=form.cleaned_data['office'])

    if form.cleaned_data['floor']:
        qs = qs.filter(floor=form.cleaned_data['floor'])

    if form.cleaned_data['city']:
        qs = qs.filter(floor__office__city=form.cleaned_data['city'])

    qs = qs.values(*ROOM_EXPORT_FIELDS)
    rooms = annotate_rooms_usage(qs)

    pages = [RoomsUsagePresenter(rooms)]

    return make_xlsx_file_response(pages, None, file_name_prefix=file_name)


def parse_staff_office_logs(import_file) -> List[dict]:
    reader = csv.DictReader(import_file.read().decode().splitlines())
    return [
        {
            'login': log['login'],
            'office_id': log['id'],
            'date': datetime.datetime.strptime(log['Дата'], '%Y-%m-%d').date(),
        }
        for log in reader
    ]


def get_map_login_to_staff(logs: List[dict]) -> Dict[str, dict]:
    list_logins = [log['login'] for log in logs]

    persons = (
        Staff
        .objects
        .filter(login__in=list_logins)
        .values('id', 'login', 'office_id', 'room_id')
    )
    return {person['login']: person for person in persons}


def get_all_logs(staff_office_logs_data: List[dict]) -> Set[tuple]:
    set_date = set([log['date'] for log in staff_office_logs_data])
    return set(
        StaffOfficeLog.objects
        .filter(date__in=set_date, source=SourceTypes.PACS.value)
        .values_list('date', 'staff_id', 'office_id')
    )


def load_staff_office_logs(staff_office_logs_data: List[dict]) -> None:
    map_login_to_staff = get_map_login_to_staff(staff_office_logs_data)
    all_logs = get_all_logs(staff_office_logs_data)

    to_create_data = set()
    for log in staff_office_logs_data:
        login = log['login']
        staff = map_login_to_staff.get(login)

        if staff is None:
            logger.error('Not found user with login: %s', login)
            continue

        staff_id, office_id, room_id = staff['id'], staff['office_id'], staff['room_id']

        if room_id is None:
            continue

        if not log['office_id'].isdigit() or office_id != int(log['office_id']):
            continue

        to_create_data.add(tuple([log['date'], staff_id, office_id]))

    logger.info('Count create staff_office_logs: %s', len(to_create_data))

    to_create_data -= all_logs
    to_create = []
    for date, staff_id, office_id in to_create_data:
        is_weekend = False
        if date.weekday() in (int(WEEKDAYS.Saturday.value), int(WEEKDAYS.Sunday.value)):
            is_weekend = True
        to_create.append(
            StaffOfficeLog(
                staff_id=staff_id,
                office_id=office_id,
                date=date,
                source=SourceTypes.PACS.value,
                is_weekend=is_weekend,
            )
        )

    StaffOfficeLog.objects.bulk_create(to_create)


@require_POST
@csrf_exempt
def staff_office_logs_import(request):
    form = FileImportForm(request.POST, request.FILES)
    if not form.is_valid():
        return JsonResponse(form_errors.sform_general_error('missing_file'), status=400)

    try:
        parsed_staff_office_logs_data = parse_staff_office_logs(request.FILES['import_file'])
    except Exception:
        return JsonResponse(form_errors.sform_general_error('incorrect_template'), status=400)

    try:
        load_staff_office_logs(parsed_staff_office_logs_data)
    except IntegrityError:
        return JsonResponse(data={}, status=409)

    return JsonResponse(data={})


CONFERENCE_ROOM_FIELDS = (
    'id',
    'num',
    'name',
    'name_exchange',
    'name_alternative',
    'floor_id',
    'room_type',
    'geometry',
    'projector',
    'panel',
    'seats',
    'marker_board',
    'cork_board',
    'video_conferencing',
    'voice_conferencing',
    'guest_wifi',
    'game_console',
    'capacity',
    'additional',
    'desk',
    'coord_x',
    'coord_y',
    'floor__office__code',
    'phone',
)


@available_for_external('map.external_with_map_access')
@responding_json
def conference_rooms(request):
    objects = Room.objects.active().filter(room_type=ROOM_TYPES.CONFERENCE)

    form = FloorFilter(request.GET or None)
    if form.is_valid() and form.cleaned_data['floor_id']:
        objects = objects.filter(floor_id=form.cleaned_data['floor_id'])
    elif form.errors:
        return {'errors': form.errors}

    result = list(objects.values(*CONFERENCE_ROOM_FIELDS))

    if workspace_management_api.user_can_manage_workspace_areas(request.user):
        _add_share_pies_to_rooms_qs_result(result)

    return result


@available_for_external('map.external_with_map_access')
@responding_json
def conference_room(request):
    form = RoomFinder(request.GET)

    if form.is_valid():
        objects = Room.objects.filter(id=form.cleaned_data['room_id'])
        room = objects[0]
    else:
        return {'errors': form.errors}, 404

    if room.room_type != ROOM_TYPES.CONFERENCE:
        return {'errors': {'room_type': ['not a conference room']}}, 404

    result = list(objects.values(*CONFERENCE_ROOM_FIELDS))
    if workspace_management_api.user_can_manage_workspace_areas(request.user):
        _add_share_pies_to_rooms_qs_result(result)

    return result[0]


@available_for_external('map.external_with_map_access')
@responding_json
def coworking_rooms(request):
    objects = Room.objects.active().filter(room_type=ROOM_TYPES.COWORKING)

    form = FloorFilter(request.GET or None)
    if form.is_valid() and form.cleaned_data['floor_id']:
        objects = objects.filter(floor_id=form.cleaned_data['floor_id'])
    elif form.errors:
        return {'errors': form.errors}

    result = list(objects.values(*ROOM_FIELDS))

    if workspace_management_api.user_can_manage_workspace_areas(request.user):
        _add_share_pies_to_rooms_qs_result(result)

    return result


@available_for_external('map.external_with_map_access')
@responding_json
def coworking_room(request):
    form = RoomFinder(request.GET)
    if form.is_valid():
        objects = Room.objects.filter(id=form.cleaned_data['room_id'])
        room = objects[0]
    else:
        return {'errors': form.errors}, 404

    if not room.room_type == ROOM_TYPES.COWORKING:
        return {'errors': {'room_type': ['not a coworking room']}}, 404

    result = list(objects.values(*ROOM_FIELDS))

    if workspace_management_api.user_can_manage_workspace_areas(request.user):
        _add_share_pies_to_rooms_qs_result(result)

    return result[0]


@available_for_external('map.external_with_map_access')
@responding_json
def special_rooms(request):
    SPECIAL_ROOM_TYPES = (
        ROOM_TYPES.ROOM_SMOKING,
        ROOM_TYPES.ROOM_COFFEE,
        ROOM_TYPES.ROOM_WC,
        ROOM_TYPES.ROOM_WARDROBE,
        ROOM_TYPES.ROOM_GYM,
        ROOM_TYPES.ROOM_LIBRARY,
        ROOM_TYPES.ROOM_SHOWER,
        ROOM_TYPES.ROOM_BALCONY,
        ROOM_TYPES.ROOM_PHONE_BOOTH,
    )
    objects = Room.objects.active().filter(room_type__in=SPECIAL_ROOM_TYPES)

    form = FloorFilter(request.GET or None)
    if form.is_valid() and form.cleaned_data['floor_id']:
        objects = objects.filter(floor_id=form.cleaned_data['floor_id'])
    elif form.errors:
        return {'errors': form.errors}

    result = list(objects.values(*ROOM_FIELDS))

    if workspace_management_api.user_can_manage_workspace_areas(request.user):
        _add_share_pies_to_rooms_qs_result(result)

    return result


EQUIPMENT_FIELDS = (
    'id',
    'floor_id',
    'name',
    'name_dns',
    'type',
    'color',
    'coord_x',
    'coord_y',
    'description',
    'floor__office__code',
    'angle',
)


@available_for_external('map.external_with_map_access')
@responding_json
def equipment(request):
    objects = Device.objects.active_without_security_devices()

    form = FloorFilter(request.GET or None)
    if form.is_valid() and form.cleaned_data['floor_id']:
        objects = objects.filter(floor_id=form.cleaned_data['floor_id'])
    elif form.errors:
        return {'errors': form.errors}

    return list(objects.values(*EQUIPMENT_FIELDS))


@available_for_external('map.external_with_map_access')
@responding_json
def single_equipment(request):
    objects = Device.objects.active_without_security_devices()

    form = EquipmentFinder(request.GET or None)
    if form.is_valid():
        objects = Device.objects.filter(id=form.cleaned_data['equipment_id'])
        # если пользователь передал id security device,
        # но у него нет права view_security_decices, то выбрасывается исключение
        for device in objects:
            if (device.type in SECURITY_DEVICES and
                    not request.user.has_perm('django_intranet_stuff.can_view_security_devices')):
                raise PermissionDenied()

    elif form.errors:
        return {'errors': form.errors}, 404
    return list(objects.values(*EQUIPMENT_FIELDS))


@available_for_external('map.external_with_map_access')
@responding_json
@permission_required('django_intranet_stuff.can_view_security_devices')
def security_devices(request):
    objects = (
        Device.objects
        .active()
        .filter(type__in=SECURITY_DEVICES)
    )

    form = FloorFilter(request.GET or None)
    if form.is_valid() and form.cleaned_data['floor_id']:
        objects = objects.filter(floor_id=form.cleaned_data['floor_id'])
    elif form.errors:
        return {'errors': form.errors}

    return list(objects.values(*EQUIPMENT_FIELDS))


REGION_FIELDS = (
    'id',
    'name',
    'description',
    'group_id',
    'group__name',
    'group__url',
    'floor_id',
    'geometry',
)


@available_for_external('map.external_with_map_access')
@responding_json
def region(request):
    form = RegionFinder(request.GET)
    if form.is_valid():
        objects = Region.objects.filter(intranet_status=1, id=form.cleaned_data['region_id'])
    else:
        return {'errors': form.errors}, 404

    region_data = objects.values(*REGION_FIELDS).get()
    centroid = region_data['geometry'].centroid
    region_data['x'] = int(centroid.x)
    region_data['y'] = int(centroid.y)
    region_data['geometry'] = RegionForm.from_polygon_to_text(region_data['geometry'])
    region_data['group'] = region_data.pop('group_id', None)
    region_data['group_name'] = region_data.pop('group__name', '')
    region_data['group_url'] = region_data.pop('group__url', '')
    return region_data


@available_for_external('map.external_with_map_access')
@responding_json
def regions(request):
    objects = Region.objects.filter(intranet_status=1)

    form = FloorFilter(request.GET or None)
    if form.is_valid() and form.cleaned_data['floor_id']:
        objects = objects.filter(floor_id=form.cleaned_data['floor_id'])
    elif form.errors:
        return {'errors': form.errors}
    regions_list = list(objects.values(*REGION_FIELDS))
    for region_data in regions_list:
        centroid = region_data['geometry'].centroid
        region_data['x'] = int(centroid.x)
        region_data['y'] = int(centroid.y)
        region_data['geometry'] = RegionForm.from_polygon_to_text(region_data['geometry'])
        region_data['group'] = region_data.pop('group_id', None)
        region_data['group_name'] = region_data.pop('group__name', '')
        region_data['group_url'] = region_data.pop('group__url', '')
    return regions_list


@responding_json
def export_edit_history(request):
    form = EditHistoryForm(request.GET or {})
    if not form.is_valid():
        return {'errors': form.errors if form.errors else 'Unknown form errors'}, 400

    logs = Log.objects.filter(
        created_at__gte=form.cleaned_data['date_from'],
        created_at__lt=form.cleaned_data['date_to'],
    ).order_by('created_at')

    return list(logs.values('who', 'model_name', 'model_pk', 'action', 'created_at'))


@require_GET
def export_rooms(request):
    pages = [RoomsExportPresenter(rooms_export_model())]
    return make_xlsx_file_response(pages, 'rooms.xlsx')
