import logging

from django.contrib.auth.decorators import permission_required
from django.views.decorators.http import require_http_methods

from staff.lib.forms.staff_form_grid import StaffFormGrid
from staff.lib.forms.errors import general_error
from staff.lib.decorators import responding_json
from staff.map.models import Room

from staff.map.forms.conference_room import ConferenceRoomForm
from staff.map.controllers import conference_room as conferenceroom_ctl

from staff.map.edit.base import load_data


logger = logging.getLogger('staff.map.edit.conference_room')


FORM_NAME = 'conference_room'
MODEL = Room


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.change_conference_room')
def edit_conferenceroom(request, conference_room_id):
    if request.method == 'GET':
        try:
            grid = StaffFormGrid(
                ConferenceRoomForm,
                initial=[conferenceroom_ctl.get_initial(conference_room_id)]
            )
        except conferenceroom_ctl.ConferenceRoomError as e:
            return {'errors': {FORM_NAME: [e.message]}}, e.status_code

        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': grid.choices_as_front_dict(),
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(request.body, ConferenceRoomForm, FORM_NAME, MODEL, conference_room_id)
    if error_code:
        return grid_data, error_code

    form_data = {'phone': ''}
    form_data.update(grid_data[0])

    try:
        conferenceroom_ctl.update_conferenceroom(conference_room_id, data=form_data, author=request.user.get_profile())
    except conferenceroom_ctl.ConferenceRoomError as e:
        return general_error({'error_key': e.message}), e.status_code

    return {'id': conference_room_id}


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.add_conference_room')
def add_conferenceroom(request):
    if request.method == 'GET':
        grid = StaffFormGrid(
            ConferenceRoomForm, initial=[conferenceroom_ctl.get_initial()])
        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': grid.choices_as_front_dict(),
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(
        request.body, ConferenceRoomForm, FORM_NAME, MODEL)
    if error_code:
        return grid_data, error_code

    form_data = {'phone': ''}
    form_data.update(grid_data[0])

    try:
        conferenceroom_instance = conferenceroom_ctl.create_conferenceroom(form_data, request.user.get_profile())
    except conferenceroom_ctl.ConferenceRoomError as e:
        return general_error({'error_key': e.message}), e.status_code

    return {'id': conferenceroom_instance.id}


@require_http_methods(['POST'])
@responding_json
@permission_required('django_intranet_stuff.delete_conference_room')
def delete_conferenceroom(request, conference_room_id):

    try:
        conferenceroom_ctl.delete_conferenceroom(
            conference_room_id, request.user.get_profile())
    except conferenceroom_ctl.ConferenceRoomError as e:
        return general_error({'error_key': e.message}), e.status_code

    return {}
