import logging

from django.contrib.auth.decorators import permission_required
from django.views.decorators.http import require_http_methods

from staff.lib.decorators import responding_json

from staff.map.models import Room

from staff.lib.forms.staff_form_grid import StaffFormGrid

from ..forms.coworking_room import CoworkingRoomForm
from ..controllers import coworking_room as ctl

from .base import load_data


logger = logging.getLogger('staff.map.edit.coworking_room')


FORM_NAME = 'coworking_room'
MODEL = Room


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.change_coworking_room')
def edit_coworkingroom(request, room_id):
    if request.method == 'GET':
        try:
            grid = StaffFormGrid(
                CoworkingRoomForm,
                initial=[ctl.get_initial(room_id)]
            )
        except ctl.CoworkingRoomError as e:
            return {'errors': {FORM_NAME: [e.message]}}, e.status_code

        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': grid.choices_as_front_dict(),
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(
        request.body, CoworkingRoomForm, FORM_NAME, MODEL, room_id)
    if error_code:
        return grid_data, error_code

    ctl.update_coworkingroom(
        room_id, data=grid_data[0], author=request.user.get_profile()
    )
    return {
        'id': room_id,
    }


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.add_coworking_room')
def add_coworkingroom(request):
    if request.method == 'GET':
        grid = StaffFormGrid(
            CoworkingRoomForm, initial=[ctl.get_initial()],
        )
        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': grid.choices_as_front_dict(),
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(request.body, CoworkingRoomForm, FORM_NAME, MODEL)
    if error_code:
        return grid_data, error_code

    try:
        instance = ctl.create_coworkingroom(
            grid_data[0], request.user.get_profile()
        )
        return {'id': instance.id}

    except ctl.CoworkingRoomError as e:
        return {'errors': {FORM_NAME: e.message}}, e.status_code


@require_http_methods(['POST'])
@responding_json
@permission_required('django_intranet_stuff.delete_coworking_room')
def delete_coworkingroom(request, room_id):
    try:
        ctl.delete_coworkingroom(room_id, request.user.get_profile())
    except ctl.CoworkingRoomError as e:
        return {'errors': {FORM_NAME: [e.message]}}, e.status_code

    return {}
