import logging

from django.contrib.auth.decorators import permission_required
from django.core.exceptions import PermissionDenied, ObjectDoesNotExist
from django.views.decorators.http import require_http_methods

from staff.lib.decorators import responding_json

from staff.map.edit.objects import EquipmentCtl
from staff.map.models import Device, SECURITY_DEVICES
from staff.map.errors import item_does_not_exist_by_id
from staff.lib.forms.staff_form_grid import StaffFormGrid

from staff.map.forms.equipment import EquipmentForm
from staff.map.controllers.equipment import (
    EquipmentError,
    get_initial,
    update_equipment,
    disable_equipment,
)
from staff.map.edit.base import load_data

logger = logging.getLogger('staff.map.edit.equipment')

FORM_NAME = 'equipment'
MODEL = Device


def _security_device_and_no_right_to_edit(type, request):
    return (
        type in SECURITY_DEVICES
        and not request.user.has_perm('django_intranet_stuff.can_add_security_devices')
    )


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.change_device')
def edit_equipment(request, equipment_id):
    if request.method == 'GET':
        try:
            initial = get_initial(equipment_id)

            if _security_device_and_no_right_to_edit(initial['type'], request):
                raise ObjectDoesNotExist('Device matching query does not exist.')

            grid = StaffFormGrid(
                EquipmentForm,
                initial=[initial]
            )
        except EquipmentError as e:
            return {'errors': {FORM_NAME: [e.message]}}, e.status_code

        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': grid.choices_as_front_dict(),
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(
        request.body, EquipmentForm, FORM_NAME, MODEL, equipment_id)

    if error_code:
        return grid_data, error_code

    if _security_device_and_no_right_to_edit(grid_data[0]['type'], request):
        raise PermissionDenied("Not enough rights to add security devices")

    with item_does_not_exist_by_id(Device)(
            logger=logger,
            message_params=[equipment_id],
            raise_e=EquipmentError
    ):
        device = Device.objects.get(id=equipment_id)

    if _security_device_and_no_right_to_edit(device.type, request):
        raise ObjectDoesNotExist('Device matching query does not exist.')

    update_equipment(device, data=grid_data[0], author=request.user.get_profile())
    return {
        'id': equipment_id,
        'is_security_device': device.type in SECURITY_DEVICES,
    }


def _try_exclude_security_devices(request, choices_types):
    if request.user.has_perm('django_intranet_stuff.can_add_security_devices'):
        return choices_types

    return [
        device_type
        for device_type in choices_types
        if device_type['value'] not in SECURITY_DEVICES
    ]


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.add_device')
def add_equipment(request):
    if request.method == 'GET':
        grid = StaffFormGrid(EquipmentForm, initial=[get_initial()])

        choices = grid.choices_as_front_dict()

        choices['type'] = _try_exclude_security_devices(request, choices['type'])

        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': choices,
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(request.body, EquipmentForm, FORM_NAME, MODEL)
    if error_code:
        return grid_data, error_code

    data = grid_data[0]

    if _security_device_and_no_right_to_edit(data['type'], request):
        raise PermissionDenied("Not enough rights to add security devices")

    try:
        equipment_instance = EquipmentCtl.create(data, request.user.get_profile())
    except Exception:
        logger.exception('Error trying to create equipment. Data: %s', data)
        return {'errors': {FORM_NAME: 'cannot-create-equipment'}}, 500

    return {
        'id': equipment_instance.id,
        'is_security_device': equipment_instance.type in SECURITY_DEVICES,
    }


@require_http_methods(['POST'])
@responding_json
@permission_required('django_intranet_stuff.delete_device')
def delete_equipment(request, equipment_id):
    try:
        with item_does_not_exist_by_id(Device)(
                logger=logger,
                message_params=[equipment_id],
                raise_e=EquipmentError
        ):
            device = Device.objects.get(id=equipment_id, intranet_status=1)

        if _security_device_and_no_right_to_edit(device.type, request):
            raise ObjectDoesNotExist('Device matching query does not exist.')

        disable_equipment(device, request.user.get_profile())
    except EquipmentError as e:
        return {'errors': {FORM_NAME: [e.message]}}, e.status_code

    return {}
