from datetime import date, datetime
from typing import Dict, Optional
from urllib.parse import urljoin as basejoin

from django.core.urlresolvers import reverse
from django.db.models import Q
from staff.person.models.contacts import ContactTypeId

from staff.person.controllers import PersonCtl
from staff.person.models import Staff

from staff.map.errors import MultipleTableError
from staff.map import handlers # noqa ignore=F401 activating receivers to store data in audit log
from staff.map.models import (
    Device,
    Room,
    ROOM_TYPES,
    Table,
    TableBook,
    TableReserve,
)
from staff.map.models.logs import LogAction, LogFactory
from staff.map.signals import (
    conference_room_updated,
    coworking_room_updated,
    equipment_updated,
    room_updated,
    table_updated,
)
from staff.map.utils import (
    check_point_within,
    find_book_conflicts,
    serialize_book,
    serialize_book_conflict,
)


class EquipmentCtl:
    def __init__(self, thing: Device, author: Staff):
        super(EquipmentCtl, self).__init__()
        self.thing: Device = thing
        self.author: Staff = author

    @property
    def url(self):
        return basejoin(reverse('map:map-home'), '#/equipment/%d' % self.thing.id)

    @classmethod
    def create(cls, data: Dict, author: Staff) -> Device:
        new_thing = Device(
            created_at=datetime.now(),
            modified_at=datetime.now(),
            from_staff_id=0,
            name=data['name'],
            name_dns=data.get('name_dns'),
            description=data.get('description'),
            coord_x=data['coord_x'],
            coord_y=data['coord_y'],
            floor_id=data['floor'].id,
            type=data['type'],
            angle=data['angle'],
        )
        new_thing.save()

        equipment_updated.send(
            sender=cls,
            obj=new_thing,
            author=author,
            action='equipment_created')

        return new_thing

    def modify(self, data):
        self.thing.modified_at = datetime.now()
        self.thing.name = data['name']
        self.thing.name_dns = data.get('name_dns')
        self.thing.description = data.get('description')
        self.thing.coord_x = data['coord_x']
        self.thing.coord_y = data['coord_y']
        self.thing.floor_id = data['floor'].id
        self.thing.type = data['type']
        self.thing.angle = data['angle']

        self.thing.save()

        equipment_updated.send(
            sender=self,
            obj=self.thing,
            author=self.author,
            action='equipment_updated')

    def disable(self):
        self.thing.intranet_status = 0
        self.thing.modified_at = datetime.now()
        self.thing.save()

        equipment_updated.send(
            sender=self,
            obj=self.thing,
            author=self.author,
            action='equipment_disabled')


class RoomCtl:
    updating_signal = room_updated
    UPDATING_ACTION = 'room_updated'

    instance_field_name = {'description': 'additional'}

    def __init__(self, room: Room, author: Staff):
        super(RoomCtl, self).__init__()
        self.instance = room
        self.author = author

    def __getattr__(self, name):
        try:
            return super(RoomCtl, self).__getattr__(name)
        except AttributeError:
            if name in self._own_fields:
                raise  # should not happened
        return getattr(self.instance, name)

    def __setattr__(self, name, value):

        is_room_attr = (
            name != 'instance'
            and hasattr(self.instance, name)
            and name not in self._own_fields
        )

        if is_room_attr:
            setattr(self.instance, name, value)
        else:
            super(RoomCtl, self).__setattr__(name, value)

    @property
    def url(self):
        return basejoin(reverse('map:map-home'), '#/room/%d' % self.instance.id)

    @property
    def _own_fields(self):
        return ['instance', 'author', 'url', 'modify', 'update', 'disable']

    @classmethod
    def create(cls, data: Dict, author: Staff):
        room = Room(
            created_at=datetime.now(),
            modified_at=datetime.now(),
            from_staff_id=0,
            coord_x=data['coord_x'],
            coord_y=data['coord_y'],
            floor_id=data['floor'].id,
            name=data['name'],
            name_en=data['name_en'],
            additional=data.get('description'),
            room_type=data['room_type'],
            num=data['num'],
        )
        room.save()
        cls.reassign_tables(room, author)
        cls.updating_signal.send(
            sender=cls,
            obj=room,
            author=author,
            action='room_created',
        )

        return cls(room, author)

    def save(self, action=None):
        action = action or self.UPDATING_ACTION
        self.instance.modified_at = datetime.now()
        self.instance.save()
        self.reassign_tables(self.instance, self.author)
        self.updating_signal.send(
            sender=self,
            obj=self.instance,
            author=self.author,
            action=action)

    @staticmethod
    def get_num_in_range(min_num, max_num, floor):
        numbers = Room.objects.active().filter(num__gt=min_num)\
                              .filter(num__lte=max_num)\
                              .filter(floor__office__city=floor.office.city)\
                              .values_list('num', flat=True)

        numbers = list(numbers)
        numbers.sort()
        suspect = min_num + 1

        for i in numbers:
            if suspect < i:
                return suspect
            suspect += 1

        if suspect < max_num:
            return suspect

    def update(self, data):
        if not isinstance(data, dict):
            return None

        for k, v in data.items():
            field_name = self.instance_field_name.get(k, k)
            if hasattr(self.instance, field_name):
                setattr(self.instance, field_name, v)

        return self

    def disable(self):
        self.instance.intranet_status = 0
        self.save(action='room_deleted')
        return self

    @classmethod
    def _update_tables(cls, room, author, table_ids=None, table_qs=None, staff_qs=None):
        if table_qs is None:
            assert table_ids is not None
            table_qs = Table.objects.active().filter(id__in=table_ids)

        if staff_qs is None:
            assert table_ids is not None
            staff_qs = Staff.objects.filter(table_id__in=table_ids)

        for table in table_qs:
            TableCtl(table, author).modify({'room': room})

        for staff in staff_qs:
            PersonCtl(staff).update({'room': room}).save(author)

    @classmethod
    def reassign_tables(cls, room: Room, author: Staff) -> None:
        """
        Привязка столов к комнате.

        Те столы, что входят в геометрию комнату будут привязаны к ней по foreign key.
        Проверяются все свободные столы, а также столы текущей комнаты на этаже.

        Те столы, что более не входят в геометрию комнаты - будут освобождены.
        """
        if room.intranet_status == 0:
            table_qs = Table.objects.active().filter(room_id=room.pk)
            staff_qs = Staff.objects.filter(room_id=room.pk)
            cls._update_tables(table_qs=table_qs, staff_qs=staff_qs, room=None, author=author)
            return

        floor = room.floor
        room_polygon = room.geom_polygon

        include_ids = []
        exclude_ids = []

        for table in Table.objects.active().filter(floor_id=floor.pk, intranet_status=1):
            # столы, привязанные к другим комнатам не трогаем
            if table.room_id and table.room.intranet_status == 1 and table.room_id != room.pk:
                continue

            table_in_room = check_point_within(table.geom_point, room_polygon)
            if table_in_room and table.room_id == room.pk:
                continue
            elif table_in_room:
                include_ids.append(table.pk)
            elif table.room_id:
                exclude_ids.append(table.pk)

        if exclude_ids:
            cls._update_tables(table_ids=exclude_ids, room=None, author=author)

        if include_ids:
            cls._update_tables(table_ids=include_ids, room=room, author=author)


class CoworkingRoomCtl(RoomCtl):
    updating_signal = coworking_room_updated
    UPDATING_ACTION = 'coworking_room_updated'

    @classmethod
    def create(cls, data: Dict, author: Staff):
        room = Room(
            created_at=datetime.now(),
            modified_at=datetime.now(),
            from_staff_id=0,
            coord_x=data['coord_x'],
            coord_y=data['coord_y'],
            geometry=data['geometry'],
            floor_id=data['floor'].id,
            name=data['name'],
            name_en=data['name_en'],
            additional=data.get('additional'),
            room_type=ROOM_TYPES.COWORKING,
            num=data['num'],
        )
        room.save()
        cls.reassign_tables(room, author)
        cls.updating_signal.send(
            sender=cls,
            obj=room,
            author=author,
            action='coworking_room_created',
        )

        return cls(room, author)

    @property
    def url(self):
        return basejoin(reverse('map:map-home'), '#/coworking_room/room_id/%d' % self.instance.id)

    def disable(self):
        self.instance.intranet_status = 0
        self.save(action='coworking_room_deleted')
        return self


class ConferenceRoomCtl(RoomCtl):
    updating_signal = conference_room_updated
    UPDATING_ACTION = 'conference_room_updated'

    @classmethod
    def create(cls, data: Dict, author: Staff):
        room = Room(
            created_at=datetime.now(),
            modified_at=datetime.now(),
            from_staff_id=0,
            room_type=ROOM_TYPES.CONFERENCE,

            coord_x=data['coord_x'],
            coord_y=data['coord_y'],
            floor_id=data['floor'].id,
            name=data['name'],
            name_en=data['name_en'],

            name_alternative=data['name_alternative'],
            name_exchange=data['name_exchange'].lower(),
            order=data['order'],
            is_cabin=data['is_cabin'],
            is_avallable_for_reserve=data['is_avallable_for_reserve'],
            geometry=data['geometry'],
            marker_board=data['marker_board'],
            cork_board=data['cork_board'],
            phone=data['phone'],
            video_conferencing=data['video_conferencing'],
            voice_conferencing=data['voice_conferencing'],
            projector=data['projector'],
            panel=data['panel'],
            desk=data['desk'],
            seats=data['seats'],
            capacity=data['capacity'],
            additional=data['additional'],
        )
        room.save()

        cls.updating_signal.send(
            sender=cls,
            obj=room,
            author=author,
            action='conference_room_created')

        return cls(room, author)

    @property
    def url(self):
        return basejoin(reverse('map:map-home'),
                        '#/conference_room/room_id/%d' % self.instance.id)

    def disable(self):
        self.instance.intranet_status = 0
        self.save(action='conference_room_deleted')
        return self


class TableCtl:
    def __init__(self, table: Optional[Table] = None, author: Optional[Staff] = None):
        super(TableCtl, self).__init__()
        self.table = table
        self.author = author

    @property
    def url(self):
        return basejoin(reverse('map:map-home'), '#/table/%d' % self.table.id)

    @property
    def occupied_by(self):
        return Staff.objects.filter(is_dismissed=False, table_id=self.table.id)

    @property
    def occupied_by_all(self):
        return Staff.objects.filter(table_id=self.table.id)

    @property
    def reserved_for(self):
        return (
            TableReserve.objects
            .active()
            .filter(table_id=self.table.id)
            .filter(intranet_status=1)
        )

    @property
    def booked_for(self):
        return TableBook.objects.filter(table_id=self.table.id, date_to__gte=date.today())

    def _get_reserve_data(self, data):
        if not data:
            return None, None

        if type(data['entity']) == Staff:
            return data['entity'], data['entity'].department
        else:
            return None, data['entity']

    def booked_for_staff(self, staff):
        """
        Для одного пользователя возвращает текущую бронь
        """
        try:
            return TableBook.objects.get(staff=staff, date_to__gte=date.today(), date_from__lte=date.today())
        except TableBook.DoesNotExist:
            return None

    @classmethod
    def reassign_table(cls, table: Table, author: Staff) -> None:
        """
        Привязка стола к комнате.

        Находим комнату для которой стол входит в геометрию комнаты.
        Проверяются все офисные комнаты и коворкинги этажа.
        """
        persons = Staff.objects.filter(table_id=table.id)
        queryset = Q(
            floor_id=table.floor.id,
            intranet_status=1,
            room_type__in=(
                ROOM_TYPES.OFFICE,
                ROOM_TYPES.COWORKING,
            ),
        )
        new_room = None
        for room in Room.objects.active().filter(queryset):
            table_in_room = check_point_within(table.geom_point, room.geom_polygon)
            if table_in_room:
                new_room = room
                break

        old_room_id = table.room_id
        new_room_id = getattr(new_room, 'id', None)
        if old_room_id == new_room_id:
            return

        table.room_id = new_room_id
        for person in persons:
            PersonCtl(person).update({'room': new_room}).save(author.user)

    @classmethod
    def create(cls, table_data: Dict, author: Staff) -> Table:
        table = Table(
            created_at=datetime.now(),
            modified_at=datetime.now(),
            from_staff_id=0,
            num=0,
            coord_x=table_data['coord_x'],
            coord_y=table_data['coord_y'],
            floor_id=table_data['floor'].id
        )
        table.save()
        cls.reassign_table(table, author)
        table.num = table.id
        table.save()
        table_updated.send(
            sender=cls,
            obj=table,
            author=author,
            action='table_created',
        )
        return table

    def add_booking(
        self,
        person: Staff,
        date_from: date,
        date_to: date,
    ) -> dict:
        assert self.table, 'Table should be set to use add_booking method'
        books_to_save = list(self.booked_for)
        book = TableBook(
            created_at=datetime.now(),
            modified_at=datetime.now(),
            staff=person,
            table=self.table,
            date_from=date_from,
            date_to=date_to,
        )

        books_to_save.append(book)
        conflicts = find_book_conflicts(books_to_save)

        if conflicts:
            errors = []
            for conflict_books, conflict_type in conflicts:
                older_book, newer_book = conflict_books
                error = {
                    'code': conflict_type,
                    'data': serialize_book_conflict(
                        conflict_books,
                        conflict_type,
                        self.table.id,
                    ),
                }

                if older_book.staff != newer_book.staff:
                    contacts = older_book.staff.contact_set
                    telegram = contacts.filter(contact_type_id=ContactTypeId.telegram.value).order_by('position')
                    error['data']['telegram'] = telegram.values_list('account_id', flat=True).first()

                errors.append(error)
            raise MultipleTableError(errors)

        book.save()

        return serialize_book(book)

    def book(self, data):
        current_books = self.booked_for
        books_to_save = []
        books_to_delete = []

        for book in current_books:
            book_data = data.pop(book.id, None)
            if book_data is None:
                books_to_delete.append(book)
            else:
                book.staff = book_data.get('staff')
                book.description = book_data.get('description')
                book.date_from = book_data.get('date_from')
                book.date_to = book_data.get('date_to')
                books_to_save.append(book)

        for book_data in data.values():

            if book_data['staff'] is None:
                continue

            book = TableBook(
                created_at=datetime.now(),
                modified_at=datetime.now(),
                staff=book_data['staff'],
                table=self.table,
                date_from=book_data['date_from'],
                date_to=book_data['date_to'],
                description=book_data.get('description', ''),
            )
            books_to_save.append(book)

        conflicts = find_book_conflicts(books_to_save)

        if conflicts:
            raise MultipleTableError([
                {
                    'error_key': conflict_type,
                    'data': serialize_book_conflict(
                        conflict_books,
                        conflict_type,
                        self.table.id,
                    ),
                }
                for conflict_books, conflict_type in conflicts
            ])

        result = {
            'book_added': [],
            'book_deleted': [],
        }

        for book in books_to_delete:
            result['book_deleted'].append(book.staff)
            book.delete()

        for book in books_to_save:
            book.save()
            result['book_added'].append(serialize_book(book))

        return result

    def occupy(self, data):
        # Если кого-то удалили - находим кого и ссаживаем
        result = {'occupy_changed': [],
                  'occupy_deleted': []}

        ctls = []

        for staff in self.occupied_by:
            staff_data = data.pop(staff.id, None)

            # Если просто стереть человека из поля, то в поле staff
            # будет пусто
            if staff_data and not staff_data['staff']:
                staff_data = None

            # Если поменяли человека, то не меняется название поля
            # TODO: починить на фронтенде, пока так.
            if staff_data and staff_data['staff'] != staff:
                data[staff_data['staff'].id] = {'staff': staff_data['staff']}
                staff_data = None

            if not staff_data:
                result['occupy_deleted'].append(staff)
                ctls.append(PersonCtl(staff).update({'table': None}))

        table_errors = []
        occupy_logs = []

        for data_item in [x for x in data.values() if x['staff']]:
            staff: Staff = data_item['staff']
            result['occupy_changed'].append((staff.table, staff))

            person_ctl = PersonCtl(staff)

            if self.table and self.table.floor.office != person_ctl.instance.office:
                table_errors.append({
                    'error_key': 'table-is-not-in-person-office',
                    'data': {
                        'person_office_name': person_ctl.instance.office and person_ctl.instance.office.name,
                        'table_office_name': self.table.floor.office and self.table.floor.office.name,
                        'table_office_id': self.table.floor.office_id,
                        'login': person_ctl.instance.login,
                    }
                })
            else:
                person_ctl.update({'table': self.table})

                # если стол привязан к комнате - привязываем к ней и пользователя
                if self.table.room_id:
                    person_ctl.update({'room': self.table.room})

                occupy_logs.append(staff.username)

            ctls.append(person_ctl)

        if table_errors:
            raise MultipleTableError(table_errors)

        for ctl in ctls:
            ctl.save(self.author.user)

        for occupy_username in occupy_logs:
            LogFactory.create_log(
                who=self.author.user,
                obj=self.table,
                action=LogAction.OCCUPIED,
                data={
                    'occupied_by': occupy_username
                }
            )

        return result

    def reserve(self, data):
        result = {'reserve_added': [],
                  'reserve_deleted': [],
                  'reserve_changed': []}

        for reserve in self.reserved_for:
            data_reserve = data.pop(reserve.id, None)
            (staff, department) = self._get_reserve_data(data_reserve)
            if not (staff or department):
                result['reserve_deleted'].append((reserve.staff, reserve.department))
                reserve.intranet_status = 0
                reserve.save()
            else:
                reserve.staff = staff
                reserve.department = department
                reserve.description = data_reserve.get('description')
                reserve.save()

        for data_item in [x for x in data.values() if x['entity']]:
            (staff, department) = self._get_reserve_data(data_item)
            new_reserve = TableReserve(
                created_at=datetime.now(),
                modified_at=datetime.now(),
                table_id=self.table.id,
                department=department,
                staff=staff,
                description=data_item.get('description', ''),
            )
            new_reserve.save()
            result['reserve_added'].append(new_reserve)

        return result

    def modify(self, data):
        self.table.modified_at = datetime.now()
        for field_name, v in data.items():
            if hasattr(self.table, field_name) and field_name != 'num':
                setattr(self.table, field_name, v)
        self.reassign_table(self.table, self.author)
        table_updated.send(
            sender=self,
            obj=self.table,
            author=self.author,
            action='table_location_updated')
        self.table.save()

    def delete(self):
        result = {'reserve_deleted': {'staff': [], 'department': []},
                  'occupy_deleted': [],
                  'book_deleted': []}
        for reserve in self.reserved_for:
            # отдельно проверять, что именно удаляется
            if reserve.staff:
                result['reserve_deleted']['staff'].append(reserve.staff)
            else:
                result['reserve_deleted']['department'].append(reserve.department)
            reserve.modified_at = datetime.now()
            reserve.intranet_status = 0
            reserve.save()
        for staff in self.occupied_by_all:
            PersonCtl(staff).update({'table': None}).save(self.author.user)
            result['occupy_deleted'].append(staff)
        for book in self.booked_for:
            result['book_deleted'].append(book.staff)
            book.delete()

        table_updated.send(
            sender=self,
            obj=self.table,
            author=self.author,
            action='table_deleted',
        )

        self.table.modified_at = datetime.now()
        self.table.intranet_status = 0
        self.table.save()

        return result
