import logging

from django.contrib.auth.decorators import permission_required
from django.views.decorators.http import require_http_methods

from staff.lib.decorators import responding_json

from staff.map.models import Region
from staff.map.forms.region import RegionForm

from staff.lib.forms.staff_form_grid import StaffFormGrid

from .base import load_data

logger = logging.getLogger(__name__)


FORM_NAME = 'region'
MODEL = Region
region_fields = ('name', 'description', 'floor', 'group', 'geometry')


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.add_room')
def add_region(request):

    if request.method == 'GET':
        grid = StaffFormGrid(RegionForm, initial=[{}],)
        choices = grid.choices_as_front_dict()
        choices.pop('group', None)
        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': choices,
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(request.body, RegionForm, FORM_NAME, MODEL)
    if error_code:
        return grid_data, error_code

    region_object = Region(floor=grid_data[0]['floor'])
    for field in region_fields:
        setattr(region_object, field, grid_data[0][field])
    region_object.save()

    return {'id': region_object.id}


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.change_room')
def edit_region(request, region_id: int):

    region_object = (
        Region.objects
        .select_related('group', 'floor__office')
        .filter(intranet_status=1, id=region_id)
        .first()
    )
    if not region_object:
        return {'errors': 'not_found'}, 404

    if request.method == 'GET':
        initial = {
            'name': region_object.name,
            'description': region_object.description,
            'group': region_object.group_id,
            'floor': region_object.floor_id,
            'geometry': RegionForm.from_polygon_to_text(region_object.geometry),
            'office': region_object.floor.office_id,
        }
        grid = StaffFormGrid(RegionForm, initial=[initial])
        choices = grid.choices_as_front_dict()
        choices.pop('group', None)
        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': choices,
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(request.body, RegionForm, FORM_NAME, MODEL, region_id)
    if error_code:
        return grid_data, error_code

    region_data = grid_data[0]

    for field in region_fields:
        setattr(region_object, field, region_data[field])
    region_object.save()

    return {
        'id': region_id,
    }


@require_http_methods(['POST'])
@responding_json
@permission_required('django_intranet_stuff.delete_room')
def delete_region(request, region_id):
    region_object = Region.objects.filter(intranet_status=1, id=region_id).first()
    if not region_object:
        return {'errors': 'not_found'}, 404
    region_object.intranet_status = 0
    region_object.save(force_update=True)
    return {}
