import json

from typing import Optional

from django.contrib.auth.decorators import permission_required
from django.views.decorators.http import require_http_methods
from django.http import HttpRequest, JsonResponse

from staff.lib.decorators import responding_json, consuming_json
from staff.lib.forms.staff_form_grid import StaffFormGrid
from staff.person.controllers import PersonCtl, Staff

from staff.map.edit.base import load_data
from staff.map.models import Room
from staff.map.forms.room import RoomForm, BindRoomForm, UnbindRoomForm
from staff.map.forms.mass_transfer import MassTransferForm
from staff.map.controllers import room as room_ctl


FORM_NAME = 'room'
MODEL = Room


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.change_room')
def edit_room(request, room_id):

    if request.method == 'GET':
        try:
            grid = StaffFormGrid(
                RoomForm,
                initial=[room_ctl.get_initial(room_id)]
            )
        except room_ctl.RoomError as e:
            return {'errors': {FORM_NAME: [e.message]}}, e.status_code

        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': grid.choices_as_front_dict(),
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(
        request.body, RoomForm, FORM_NAME, MODEL, room_id)
    if error_code:
        return grid_data, error_code

    room_ctl.update_room(
        room_id, data=grid_data[0], author=request.user.get_profile()
    )
    return {
        'id': room_id,
    }


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.add_room')
def add_room(request):

    if request.method == 'GET':
        try:
            grid = StaffFormGrid(
                RoomForm,
                initial=[
                    room_ctl.get_initial(floor_id=request.GET.get('floor_id'))
                ],
            )
            return {
                'floor_office_mapping': grid.form_class.floor_office_mapping(),
                'choices': grid.choices_as_front_dict(),
                FORM_NAME: grid.as_front_dict(),
            }

        except room_ctl.FloorError as e:
            return {'errors': {FORM_NAME: e.message}}, e.status_code

    grid_data, error_code = load_data(request.body, RoomForm, FORM_NAME, MODEL)
    if error_code:
        return grid_data, error_code

    try:
        room_instance = room_ctl.create_room(
            grid_data[0], request.user.get_profile()
        )
    except room_ctl.RoomError as e:
        return {'errors': {FORM_NAME: e.message}}, e.status_code

    return {'id': room_instance.id}


@require_http_methods(['POST'])
@responding_json
@permission_required('django_intranet_stuff.delete_room')
def delete_room(request, room_id):

    try:
        room_ctl.delete_room(room_id, request.user.get_profile())
    except room_ctl.RoomError as e:
        return {'errors': {FORM_NAME: [e.message]}}, e.status_code

    return {}


def change_person_room(staff: Staff, room: Optional[Room], author: Staff) -> None:
    """
    Изменение комнаты у сотрудника с отсаживанием со стола
    """
    person_ctl = PersonCtl(staff)
    person_ctl.update(
        {
            'table': None,
            'room': room,
        },
    ).save(author)


@require_http_methods(['POST'])
@consuming_json
def bind_room(request: HttpRequest, room_id: int, json_data: dict) -> JsonResponse:
    json_data['room'] = room_id
    form = BindRoomForm(data=json_data)
    if not form.is_valid():
        return JsonResponse(form.errors, status=400)

    staff = form.cleaned_data['person']
    room = form.cleaned_data['room']
    change_person_room(
        staff=staff,
        room=room,
        author=request.user,
    )
    return JsonResponse({}, status=200)


@require_http_methods(['POST'])
@consuming_json
def unbind_room(request: HttpRequest, json_data: dict) -> JsonResponse:
    form = UnbindRoomForm(data=json_data)
    if not form.is_valid():
        return JsonResponse(form.errors, status=400)

    change_person_room(
        staff=form.cleaned_data['person'],
        room=None,
        author=request.user,
    )
    return JsonResponse({}, status=200)


@require_http_methods(['POST'])
def mass_workplace_transfer(request: HttpRequest) -> JsonResponse:
    data = json.loads(request.body)
    form = MassTransferForm(data)

    if not form.is_valid():
        return JsonResponse(data=form.errors_as_dict(), status=400)

    staff_with_other_office = []

    for person in form.cleaned_data['persons']:
        if person.office != form.cleaned_data['room'].floor.office:
            staff_with_other_office.append(person.login)
        else:
            change_person_room(person, form.cleaned_data['room'], request.user)

    if staff_with_other_office:
        error_data = {
            'errors': {
                'persons': [
                    {
                        'code': 'can-not-transfer-btw-offices',
                        'params': {
                            'new_office_id': form.cleaned_data['room'].floor.office.id,
                            'logins': staff_with_other_office,
                        },
                    },
                ],
            },
        }
        return JsonResponse(error_data, status=400)

    return JsonResponse({}, status=200)
