import json
import logging

from django.contrib.auth.decorators import permission_required
from django.http import JsonResponse
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods, require_POST

from staff.lib.decorators import responding_json, auth_by_tvm_only
from staff.lib.forms.errors import invalid_json_error
from staff.lib.forms.staff_form_grid import StaffFormGrid

from staff.map.controllers import table as table_ctl
from staff.map.controllers.table import book_table_for_person_controller
from staff.map.edit.base import load_data
from staff.map.forms.book_table_for_person import BookTableForPersonForm
from staff.map.forms.table import (
    BookTableForm,
    OccupyTableForm,
    ReserveTableForm,
    TableForm,
)
from staff.map.models import Table


logger = logging.getLogger('staff.map.edit.table')

FORM_NAME = 'table'
MODEL = Table


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.change_table')
def edit_table(request, table_id):
    if request.method == 'GET':
        try:
            grid = StaffFormGrid(
                TableForm,
                initial=[table_ctl.get_initial(table_id)]
            )
        except table_ctl.TableError as e:
            return {'errors': {FORM_NAME: [e.message]}}, e.status_code

        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': grid.choices_as_front_dict(),
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(request.body, TableForm, FORM_NAME, MODEL, table_id)
    if error_code:
        return grid_data, error_code

    table_ctl.update_table(
        table_id, data=grid_data[0], author=request.user.get_profile()
    )
    return {'id': table_id}


@require_http_methods(['GET', 'POST'])
@responding_json
@permission_required('django_intranet_stuff.add_table')
def add_table(request):
    if request.method == 'GET':
        grid = StaffFormGrid(TableForm, initial=[table_ctl.get_initial()])
        return {
            'floor_office_mapping': grid.form_class.floor_office_mapping(),
            'choices': grid.choices_as_front_dict(),
            FORM_NAME: grid.as_front_dict(),
        }

    grid_data, error_code = load_data(request.body, TableForm, FORM_NAME, MODEL)
    if error_code:
        return grid_data, error_code

    try:
        table_instance = table_ctl.create_table(
            grid_data[0], request.user.get_profile()
        )
    except table_ctl.TableError as e:
        return {'errors': {FORM_NAME: e.message}}, e.status_code

    return {'id': table_instance.id}


@require_http_methods(['POST'])
@responding_json
@permission_required('django_intranet_stuff.delete_table')
def delete_table(request, table_id):
    try:
        return table_ctl.delete_table(table_id, request.user.get_profile())
    except table_ctl.TableError as e:
        return {'errors': {FORM_NAME: [e.message]}}, e.status_code


@require_http_methods(['POST', 'GET'])
@responding_json
@permission_required('django_intranet_stuff.change_table_occupation')
def occupy_table(request, table_id):

    if request.method == 'GET':
        return table_ctl.get_initial_occupy(table_id)

    first_level_forms = {
        'occupation': OccupyTableForm,
        'reserve': ReserveTableForm,
        'book': BookTableForm,
    }
    request_body = request.body
    try:
        occupation_data = json.loads(request_body)
    except ValueError:
        logger.exception('Wrong JSON: %s', request_body)
        return invalid_json_error(request_body), 400

    grids = {
        form_name: StaffFormGrid(
            form_class, data=occupation_data.get(form_name, {}))
        for form_name, form_class in first_level_forms.items()
    }

    errors = {
        grid_type: grid.errors()
        for grid_type, grid in grids.items()
        if not grid.is_valid()
    }

    if errors:
        return {'errors': errors}, 400
    try:
        occupation_result = table_ctl.occupy(table_id, occupation_data=grids, author=request.user.get_profile())
    except table_ctl.MultipleTableError as e:
        return {'errors': {FORM_NAME: [{'office': [error for error in e.errors]}]}}, 400
    except table_ctl.TableError as e:
        return {'errors': {FORM_NAME: e.message}}, e.status_code
    return occupation_result


@csrf_exempt
@require_POST
@auth_by_tvm_only(['staff', 'crmhd', 'chat_bot'])
def book_table_for_person(request):
    try:
        form = BookTableForPersonForm(json.loads(request.body))
    except ValueError:
        logger.info('Invalid JSON received', exc_info=True)
        return JsonResponse({}, status=400)

    if not form.is_valid():
        return JsonResponse(data=form.errors_as_dict(), status=400)

    try:
        result = book_table_for_person_controller(
            form.cleaned_data['person'],
            form.cleaned_data['table'],
            form.cleaned_data['date_from'],
            form.cleaned_data['date_to'],
        )
    except table_ctl.MultipleTableError as e:
        return JsonResponse(data={'errors': [error for error in e.errors]}, status=400)

    return JsonResponse(data=result, status=200)
