import re

from staff.map.models import Device, Office, Floor, Room, Table

from staff.lib.utils.qs_values import localize
from staff.lib.forms.staff_form import StaffForm
from staff.lib.forms.staff_fields import (
    StaffCharField,
    StaffIntegerField,
    StaffLocalizedModelChoiceField,
)


class BaseMapEditForm(StaffForm):

    office = StaffLocalizedModelChoiceField(
        queryset=Office.objects.filter(
            have_map=True, intranet_status=1).order_by('position'),
        empty_label=None,
        required=False,
    )
    floor = StaffLocalizedModelChoiceField(
        queryset=Floor.objects.filter(
            office__have_map=True,
            office__intranet_status=1,
            intranet_status=1,
        ).order_by('position'),
        empty_label=None,
        required=True,
    )
    coord_x = StaffIntegerField(required=True)
    coord_y = StaffIntegerField(required=True)

    def __init__(self, *args, **kwargs):
        instance_data = kwargs.get('data', {})
        self.id = instance_data.pop('id', None)
        self.model = instance_data.pop('model', None)
        super(BaseMapEditForm, self).__init__(*args, **kwargs)

    @staticmethod
    def floor_office_mapping():
        mapping = {}
        floors_data = (
            Floor.objects
            .values_list('office_id', 'id', 'name', 'name_en')
            .filter(intranet_status=1)
            .order_by('office', 'num')
        )
        for office_id, floor_id, name, name_en in floors_data:

            localized = localize({
                'floor_id': floor_id,
                'name': name,
                'name_en': name_en,
            })
            mapping.setdefault(office_id, []).append({
                'value': localized['floor_id'],
                'label': localized['name'],
            })
        return mapping

    def clean(self):
        floor = self.cleaned_data.get('floor')

        coord_x = self.cleaned_data.get('coord_x')
        coord_y = self.cleaned_data.get('coord_y')

        entities = [Room, Table, Device, ]

        queries = (
            model.objects.active().filter(
                coord_x=coord_x, coord_y=coord_y, floor=floor)
            for model in entities
        )
        if self.id is not None:
            queries = (
                qs.exclude(id=self.id)
                if qs.model == self.model else qs
                for qs in queries
            )

        if any((qs.exists() for qs in queries)):
            msg = ['{"error_key": "map-coordinates_occupied"}']
            self._errors['coord_x'] = msg
            self._errors['coord_y'] = msg

        return self.cleaned_data


GEOMETRY_VALIDATION_PATTERN = re.compile(r'^-?\d+(?:,-?\d+)+$')
GEOMETRY_EXTRACTION_PATTERN = re.compile(r'-?\d+')


class GeometryBaseMapEditForm(BaseMapEditForm):
    geometry = StaffCharField(max_length=2055, required=False)

    def clean(self):
        cleaned_data = super(GeometryBaseMapEditForm, self).clean()
        geometry = cleaned_data.get('geometry')

        if geometry:
            if GEOMETRY_VALIDATION_PATTERN.match(geometry) is None:
                self._errors['geometry'] = ['{"error_key": "default-field-invalid"}']
            else:
                digits = GEOMETRY_EXTRACTION_PATTERN.findall(geometry)
                if len(digits) < 8 or len(digits) % 2 != 0 or digits[0] != digits[-2] or digits[1] != digits[-1]:
                    self._errors['geometry'] = ['{"error_key": "geometry-contains-non-closed-rings"}']

        return cleaned_data
