from typing import List

import sform

from django.conf import settings
from django.core.exceptions import ValidationError, ObjectDoesNotExist

from staff.person.models import Staff

from staff.map.models import Room


class MassTransferForm(sform.SForm):

    room = sform.IntegerField()
    persons = sform.GridField(
        sform.CharField(max_length=150),
        state=sform.REQUIRED,
    )

    @staticmethod
    def clean_room(value: int) -> Room:
        try:
            room = Room.objects.get(id=value, intranet_status=1)
        except ObjectDoesNotExist:
            raise ValidationError('Room does not exist', code='room-does-not-exist')

        return room

    @staticmethod
    def clean_persons(value: List[str]) -> List[Staff]:
        if len(value) != len(set(value)):
            raise ValidationError('Logins must be unique', code='logins-must-be-unique')

        if len(value) > settings.MAX_PERSONS_FOR_WORKPLACE_TRANSFER:
            raise ValidationError(
                'Too much logins',
                code='too-much-logins',
                params={
                    'max_logins_amount': settings.MAX_PERSONS_FOR_WORKPLACE_TRANSFER,
                },
            )

        persons = Staff.objects.active().filter(login__in=value)

        return persons
