from itertools import chain

from django.contrib.gis.geos import Polygon, MultiPolygon, GEOSException
from django.core.exceptions import ValidationError

from staff.groups.models import Group, GROUP_TYPE_CHOICES


from staff.lib.forms.staff_fields import (
    StaffCharField,
    StaffModelChoiceField,
    StaffIntegerField,
)

from staff.map.forms.base import GeometryBaseMapEditForm


class RegionForm(GeometryBaseMapEditForm):

    coord_x = StaffIntegerField(required=False)
    coord_y = StaffIntegerField(required=False)

    name = StaffCharField(max_length=255, required=False)
    description = StaffCharField(required=False)

    group = StaffModelChoiceField(
        queryset=Group.objects.filter(
            intranet_status=1,
            type=GROUP_TYPE_CHOICES.SERVICE,
        ),
        empty_label=None,
        required=True,
        label='name'
    )

    def clean(self):
        cleaned_data = super(RegionForm, self).clean()
        cleaned_data['geometry'] = self.from_text_to_polygon(cleaned_data['geometry'])

        return cleaned_data

    def as_front_dict(self):
        data = super().as_front_dict()
        group_id = data['group']['value'] or None
        group_name, group_url = (
            Group.objects
            .values_list('name', 'url')
            .filter(id=group_id)
            .last()
        ) if group_id else ('', '')

        data['group']['group_name'] = group_name
        data['group']['group_url'] = group_url

        return data

    @staticmethod
    def from_text_to_polygon(text_value: str) -> MultiPolygon:
        try:
            coordinates = [int(v) for v in text_value.split(',')]
        except ValueError:
            raise ValidationError(code='wrong_number_in_coordinates')

        if len(coordinates) % 2 != 0:
            raise ValidationError(code='odd_coordinate_numbers')

        points = list(zip(coordinates[::2], coordinates[1::2]))
        points.append(points[0])

        try:
            polygon = Polygon(points)
        except GEOSException:
            raise ValidationError(code='geos_exception')

        return MultiPolygon(polygon)

    @staticmethod
    def from_polygon_to_text(multipolygon_value: MultiPolygon) -> str:
        first_polygon = multipolygon_value[0]
        ext_ring = first_polygon[0]
        str_coordinates = [str(int(v)) for v in chain.from_iterable(ext_ring[:-1])]
        return ','.join(str_coordinates)
