import sform
from django.core.exceptions import ValidationError

from staff.lib.forms.staff_fields import (
    StaffCharField,
    StaffChoiceField,
    StaffIntegerField,
)
from staff.person.models import Staff

from staff.map.models import Room, ROOM_TYPES
from staff.map.forms.base import GeometryBaseMapEditForm


class RoomForm(GeometryBaseMapEditForm):

    num = StaffIntegerField(required=False)
    room_type = StaffChoiceField(choices=ROOM_TYPES.choices())

    name = StaffCharField(max_length=255, required=False)
    name_en = StaffCharField(max_length=255, required=False)

    additional = StaffCharField(required=False)

    def choices_as_front_dict(self):
        choices = super(RoomForm, self).choices_as_front_dict()
        choices['room_type'] = [
            rt for rt in choices['room_type']
            if rt['value'] not in [ROOM_TYPES.CONFERENCE, ROOM_TYPES.COWORKING]
        ]
        return choices

    def clean(self):
        cleaned_data = super(RoomForm, self).clean()
        num = cleaned_data.get('num')
        floor = cleaned_data.get('floor')

        if num and floor:
            query = Room.objects.filter(
                floor__office_id=floor.office_id,
                floor__intranet_status=1,
                num=num, intranet_status=1
            )
            if self.id:
                query = query.exclude(id=self.id)
                if self.id is not None:
                    query = query.exclude(id=self.id)

                if query.exists():
                    self._errors['num'] = ['{"error_key": "map-room_duplicate"}']

        return cleaned_data


class BindRoomForm(sform.SForm):

    person = sform.ModelChoiceField(
        state=sform.REQUIRED,
        empty_label=None,
        queryset=Staff.objects.filter(
            is_dismissed=False,
            is_robot=False,
        ),
        to_field_name='login',
        label_extractor='login',
    )

    room = sform.ModelChoiceField(
        state=sform.REQUIRED,
        empty_label=None,
        queryset=Room.objects.select_related('floor'),
        label_extractor='id',
    )

    def _validate_office(self, current_office_id, requested_office_id):
        if requested_office_id != current_office_id:
            raise ValidationError(
                message='Requested room is in foreign office',
                code='foreign-office',
                params={'office_id': requested_office_id}
            )

    def clean(self):
        cleaned_data = self.cleaned_data
        if not self.is_valid():
            return cleaned_data
        current_office_id = self.cleaned_data['person'].office_id
        requested_office_id = self.cleaned_data['room'].floor.office_id
        self._validate_office(current_office_id, requested_office_id)
        return cleaned_data


class UnbindRoomForm(sform.SForm):

    person = sform.ModelChoiceField(
        state=sform.REQUIRED,
        empty_label=None,
        queryset=Staff.objects.filter(
            is_dismissed=False,
            is_robot=False,
        ),
        to_field_name='login',
        label_extractor='id',
    )
