from datetime import timedelta, date
from typing import Optional

from django.conf import settings
from django.core.exceptions import ValidationError

from staff.lib.forms.staff_fields import (
    StaffCharField,
    StaffDateField,
    StaffIntegerField,
    StaffSuggestField,
)
from staff.lib.forms.staff_form import StaffForm
from staff.person.models import Staff

from staff.map.forms.base import BaseMapEditForm


class TableForm(BaseMapEditForm):
    num = StaffIntegerField(required=False)


class BaseTableReserveForm(StaffForm):
    person_field_name: Optional[str] = None

    person = StaffSuggestField(required=False)

    def clean_person(self):
        person_id = self.cleaned_data.get('person')
        if not person_id:
            return
        try:
            return (
                Staff.objects
                .filter(is_dismissed=False, is_robot=False)
                .get(id=person_id)
            )
        except Staff.DoesNotExist:
            raise ValidationError(
                StaffSuggestField.default_error_messages['invalid']
            )

    def __init__(self, data=None, *args, **kwargs):
        super(BaseTableReserveForm, self).__init__(data=data, *args, **kwargs)

    def clean(self):
        cleaned_data = self.cleaned_data
        if 'person' in cleaned_data:
            cleaned_data[self.person_field_name] = cleaned_data.pop('person')
        return cleaned_data


class OccupyTableForm(BaseTableReserveForm):
    person_field_name = 'staff'


class ReserveTableForm(BaseTableReserveForm):
    person_field_name = 'entity'

    description = StaffCharField(max_length=255, required=False)


class BookTableForm(BaseTableReserveForm):
    person_field_name = 'staff'

    date_from = StaffDateField(required=False)
    date_to = StaffDateField(required=False)
    description = StaffCharField(max_length=255, required=False)

    def _date_required(self, date_field_name):
        cleaned_data = self.cleaned_data
        if 'person' in cleaned_data and cleaned_data['person']:
            if not cleaned_data[date_field_name]:
                raise ValidationError(StaffDateField.default_error_messages['invalid'])
        return cleaned_data[date_field_name]

    def clean_date_from(self):
        return self._date_required('date_from')

    def clean_date_to(self):
        return self._date_required('date_to')

    def clean(self):
        cleaned_data = self.cleaned_data
        if 'person' in cleaned_data:
            cleaned_data[self.person_field_name] = cleaned_data.pop('person')
        if (
            cleaned_data.get('date_to') is not None
            and cleaned_data.get('date_from') is not None
        ):
            if cleaned_data['date_to'] < cleaned_data['date_from']:
                raise ValidationError(StaffDateField.default_error_messages['invalid'])
            elif cleaned_data['date_to'] - date.today() > timedelta(settings.MAP_TABLE_MAX_BOOK_PERIOD_FROM_TODAY):
                raise ValidationError('{"error_key": "two-weeks-exceeded"}')
        return cleaned_data
