from typing import Dict

from django.core.management.base import BaseCommand
from django.db.models.query import Prefetch

from staff.lib.db import atomic
from staff.map.models import Room, Table, ROOM_TYPES, Floor, Office
from staff.map.utils import check_point_within
from staff.person.models import Staff


class Command(BaseCommand):
    help = "Find table in the room geometry"

    _tables: Dict[int, int] = {}
    _cached_tables = None

    def add_arguments(self, parser):
        parser.add_argument('office_id', nargs='?', help='Office ID')
        parser.add_argument('floor_id', nargs='?', help='Floor ID')
        parser.add_argument('--room', dest='room_id', help='Room ID')
        parser.add_argument('--update-user-room', action='store_true', dest='update_user_room', default=False)
        parser.add_argument('--skip-assigned', action='store_true', dest='skip_assigned', default=False)
        parser.add_argument('--dry-run', action='store_true', dest='dry_run', default=False)
        parser.add_argument('--all-offices', action='store_true', dest='all_offices', default=False)

    def get_room(self, pk):
        try:
            return Room.objects.get(
                pk=pk,
                room_type__in=(
                    ROOM_TYPES.OFFICE,
                    ROOM_TYPES.COWORKING,
                ),
            )
        except Room.DoesNotExist:
            self.stdout.write(f"Room {pk} not found!")

    def get_floor(self, pk):
        try:
            return Floor.objects.get(pk=pk)
        except Floor.DoesNotExist:
            self.stdout.write(f"Floor {pk} not found!")

    def prefetch_active_floors(self):
        return Prefetch('floor_set', Floor.objects.filter(intranet_status=1), to_attr='active_floors')

    def get_office(self, pk):
        prefetch = self.prefetch_active_floors()
        return Office.objects.filter(pk=pk).prefetch_related(prefetch).first()

    def get_all_tables(self, floor: Floor):
        if not self._cached_tables:
            self._cached_tables = Table.objects.filter(floor_id=floor.pk, intranet_status=1)

        return self._cached_tables

    def get_all_offices(self):
        prefetch = self.prefetch_active_floors()
        return Office.objects.filter(intranet_status=1).prefetch_related(prefetch)

    def clear_table_cache(self) -> None:
        self._cached_tables = None
        self._tables = {}

    def update_tables(self, table_ids, room_id=None) -> None:
        if self.dry_run:
            return

        self.stdout.write('Updating tables...')
        Table.objects.filter(id__in=table_ids).update(room_id=room_id)
        if self.update_user_room:
            self.stdout.write('Updating user rooms...')
            Staff.objects.filter(table_id__in=table_ids).update(room_id=room_id)

    def check_room_tables(self, room: Room):
        self.stdout.write(f"Room: {room} {room.pk}")

        include_ids = []
        exclude_ids = []
        for table in self.get_all_tables(room.floor):
            # пропускаем столы, которые привязаны к другим комнатам
            if self.skip_assigned and (table.room_id and table.room_id != room.pk):
                continue

            is_in = check_point_within(table.geom_point, room.geom_polygon)
            if not is_in:
                # если стол был ранее привязан - его нужно отвязать от комнаты
                if table.room_id:
                    exclude_ids.append(table.pk)
                continue

            if table.pk not in self._tables:
                self._tables[table.pk] = room.pk
                include_ids.append(table.pk)
                self.stdout.write(f"Table: {table}")
            else:
                self.stdout.write(f"Table: {table} already used in room {self._tables[table.pk]}")

        if exclude_ids:
            self.stdout.write(f"Unassign tables from the room: {exclude_ids}")
            self.update_tables(exclude_ids, room_id=None)

        if include_ids:
            self.stdout.write(f"Assign tables to the room: {include_ids}")
            self.update_tables(include_ids, room_id=room.pk)

        if not exclude_ids and not include_ids:
            self.stdout.write("No table found")

    def check_floor_rooms(self, floor: Floor):
        self.stdout.write(f"Floor: {floor}")

        rooms = floor.room_set.filter(
            room_type__in=(
                ROOM_TYPES.OFFICE,
                ROOM_TYPES.COWORKING,
            ),
            geometry__isnull=False,
            intranet_status=1,
        )
        for room in rooms:
            self.check_room_tables(room)

        self.clear_table_cache()

    def check_office_floors(self, office: Office):
        self.stdout.write(f"Office: {office} {office.pk}")

        floors = office.active_floors
        for floor in floors:
            self.check_floor_rooms(floor)

    def handle(self, *args, **options):
        self.update_user_room = options.get('update_user_room', False)
        self.skip_assigned = options.get('skip_assigned', False)
        self.dry_run = options.get('dry_run', False)

        office_id = options.get('office_id')
        floor_id = options.get('floor_id')
        room_id = options.get('room_id')
        all_offices = options.get('all_offices', False)

        started = False
        with atomic():
            if room_id:
                room = self.get_room(room_id)
                if room:
                    started = True
                    self.check_room_tables(room)

            elif floor_id:
                floor = self.get_floor(floor_id)
                if floor:
                    started = True
                    self.check_floor_rooms(floor)

            elif office_id:
                office = self.get_office(office_id)
                if office:
                    started = True
                    self.check_office_floors(office)
            elif all_offices:
                started = True
                for office in self.get_all_offices():
                    self.check_office_floors(office)

        if started:
            self.stdout.write("Done")
