from typing import Optional

from django.db import models
from django.utils.translation import ugettext_lazy as _

from staff.lib.models.base import IntranetModel
from staff.lib.utils.ordered_choices import OrderedChoices

from staff.map.models.utils import geom_to_polygon
from staff.map.models.office import SourceTypes


ROOM_TYPES = OrderedChoices(
    ('OFFICE', 0, _('intranet_stuff.Room_Office')),
    ('CONFERENCE', 1, _('intranet_stuff.Room_ConferenceRoom')),
    ('ROOM_SMOKING', 2, _('intranet_stuff.Room_SmokingArea')),
    ('ROOM_COFFEE', 3, _('intranet_stuff.Room_CoffeePoint')),
    ('ROOM_WC', 4, _('intranet_stuff.Room_WC')),
    ('ROOM_WARDROBE', 5, _('intranet_stuff.Room_Wardrobe')),
    ('ROOM_WAREHOUSE', 6, _('intranet_stuff.Room_Warehouse')),
    ('ROOM_SERVICE', 7, _('intranet_stuff.Room_StaffOnly')),
    ('ROOM_SHOWER', 8, _('intranet_stuff.Room_Shower')),
    ('ROOM_GYM', 9, _('intranet_stuff.Room_Gym')),
    ('ROOM_LIBRARY', 10, _('intranet_stuff.Room_Library')),
    ('COWORKING', 11, _('intranet_stuff.Room_Coworking')),
    ('ROOM_STAIRS', 12, _('intranet_stuff.Room_Stairs')),
    ('ROOM_SWITCH', 13, _('intranet_stuff.Room_Switch')),
    ('ROOM_HALLWAY', 14, _('intranet_stuff.Room_Hallway')),
    ('ROOM_BALCONY', 15, _('intranet_stuff.Room_Balcony')),
    ('ROOM_PHONE_BOOTH', 16, _('intranet_stuff.Room_PhoneBooth')),
)


ROOM_LABEL_POS_CHOICES = (
    ('T', 'над помещением'),
    ('B', 'под помещением'),
)


ROOM_ISO_LABEL_CHOICES = (
    ('', 'снизу в центре'),
    ('br', 'снизу слева'),
    ('bl', 'снизу справа'),
    ('r', 'слева'),
    ('l', 'справа'),
)


class Room(IntranetModel):
    from_staff_id = models.PositiveIntegerField(db_index=True, null=True, blank=True)

    floor = models.ForeignKey('Floor')

    num = models.PositiveIntegerField(null=True, blank=True, verbose_name='Номер комнаты')

    name = models.CharField(max_length=255, default='')
    name_en = models.CharField(max_length=255, default='')
    name_exchange = models.CharField(max_length=50, default='', db_index=True)
    name_alternative = models.CharField(max_length=255, default='')

    room_type = models.PositiveSmallIntegerField(choices=ROOM_TYPES.choices(), default=ROOM_TYPES.CONFERENCE)

    coord_x = models.IntegerField(default=0)
    coord_y = models.IntegerField(default=0)

    color = models.CharField(max_length=10, default='')

    geometry = models.CharField(max_length=2055, default='')

    phone = models.CharField(max_length=50, default='', verbose_name='номер телефона', blank=True)

    iso_x = models.IntegerField(blank=True, null=True, verbose_name='координата X для киоска')
    iso_y = models.IntegerField(blank=True, null=True, verbose_name='координата Y для киоска')
    iso_label_pos = models.CharField(choices=ROOM_ISO_LABEL_CHOICES, max_length=2,
                                     blank=True, default="",
                                     verbose_name='Положение указателя для киоска')
    hide_floor_num = models.BooleanField(verbose_name='Показывать без номера этажа в киосках', default=False)

    order = models.PositiveSmallIntegerField(blank=True, null=True, verbose_name='порядок')
    label_pos = models.CharField(choices=ROOM_LABEL_POS_CHOICES, max_length=1,
                                 blank=False, default='T',
                                 verbose_name='Подпись переговорки')

    projector = models.PositiveSmallIntegerField(verbose_name='проектор', blank=True, null=True, help_text='количество')
    panel = models.PositiveSmallIntegerField(verbose_name='ЖК/Плазменная панель',
                                             blank=True, null=True,
                                             help_text='количество')
    seats = models.PositiveSmallIntegerField(verbose_name='число посадочных мест', blank=True, null=True)
    marker_board = models.BooleanField(verbose_name='маркерная доска', blank=True, default=False)
    cork_board = models.BooleanField(verbose_name='пробковая доска', blank=True, default=False)
    video_conferencing = models.CharField(max_length=10, verbose_name='видео конференц-связь',
                                          blank=True, null=True, help_text='номер')
    voice_conferencing = models.BooleanField(verbose_name='голосовая конференц-связь', blank=True, default=False)
    game_console = models.CharField(verbose_name='игровая приставка', max_length=255,
                                    blank=True, null=True, help_text='если есть, то какая')
    capacity = models.CharField(verbose_name='вместимость', max_length=255, blank=True, null=True,
                                help_text='оценка числа человек')
    additional = models.TextField(verbose_name='дополнительная информация', blank=True, null=True)
    desk = models.BooleanField(verbose_name='стол', default=False, blank=True)

    is_cabin = models.BooleanField(verbose_name='Это домик', default=False)

    is_avallable_for_reserve = models.BooleanField(verbose_name='Доступна для бронирования', default=False)

    guest_wifi = models.BooleanField(verbose_name='Гостевой Wi-Fi', default=False)

    def __str__(self):
        return '%s %s' % (self.num, self.name)

    @property
    def geom_polygon(self) -> Optional[str]:
        return geom_to_polygon(self.geometry)

    class Meta(IntranetModel.Meta):
        db_table = 'intranet_room'
        permissions = (
            ('add_conference_room', 'Can add conference room'),
            ('change_conference_room', 'Can change conference room'),
            ('delete_conference_room', 'Can delete conference room'),
            ('add_coworking_room', 'Can add coworking room'),
            ('change_coworking_room', 'Can change coworking room'),
            ('delete_coworking_room', 'Can delete coworking room'),
        )


class RoomUsage(models.Model):
    room = models.ForeignKey(Room, verbose_name='комната')
    date = models.DateField()
    usage = models.FloatField()
    square_based_usage = models.FloatField(null=True, default=None)
    source = models.CharField(
        null=False,
        blank=False,
        max_length=16,
        choices=SourceTypes.choices(),
    )
