import csv
import logging
from datetime import timedelta, date, datetime
from io import BytesIO
from typing import Optional

from django.conf import settings
from django.core import mail
from django.db import IntegrityError
from django.db.models import F, Count, Q
from django.utils import timezone
from PIL import Image

from staff.celery_app import app
from staff.lib.tasks import LockedTask
from staff.lib import requests
from staff.whistlah.models import StaffLastOffice

from staff.map.api.views import load_staff_office_logs
from staff.map.models import (
    FloorMap,
    StaffOfficeLog,
    SourceTypes,
    RoomUsage,
    ROOM_TYPES,
    Room,
)
from staff.map.storage import fetch_file_from_storage, save_file_to_storage
from staff.map.tiles import slice_map
from staff.map.utils import get_map_room_id_to_room_square

TARGET_URL = 'https://%s/map/tiles/{0}' % settings.STAFF_HOST

logger = logging.getLogger(__name__)
# отключаю логи boto3 и botocore
logging.getLogger('boto3').propagate = False
logging.getLogger('botocore').propagate = False


def save_image_to_mds(floormap_filename):
    response = requests.get(TARGET_URL.format(floormap_filename))
    temp_file = BytesIO(response.content)
    temp_file.seek(0)
    save_file_to_storage(floormap_filename, temp_file)


@app.task(ignore_result=True)
class TileCutter(LockedTask):
    name_template = '{file_name}-{z}-{x}-{y}'

    def locked_run(self, map_id, confirm_email, map_is_for, *args, **kwargs):
        logger.info('Starting tilecutter for floormap with id %s', str(map_id))
        try:
            self.floor_map = FloorMap.objects.get(id=map_id)
            self.cut()

            self.floor_map.is_ready = True
            self.floor_map.save()

            self._send_success_message(confirm_email, map_is_for)
            logger.info('Tiles successfully saved for floormap with id %s', str(map_id))
        except Exception:
            self._send_fail_message(confirm_email, map_is_for)
            logger.exception('Can\'t cut map for floormap with id %s', str(map_id))

    def _send_success_message(self, confirm_email, map_is_for):
        body = 'Обновление карты \'{map_is_for}\' успешно завершено'.format(
            map_is_for=map_is_for,
        )

        self._send_message(confirm_email, body)

    def _send_message(self, confirm_email, body):
        message = mail.EmailMessage(
            subject='Обновление карты',
            from_email=settings.ROBOT_STAFF_LOGIN + '@yandex-team.ru',
            to=[confirm_email],
            body=body,
            connection=mail.get_connection()
        )
        message.send()

    def _send_fail_message(self, confirm_email, map_is_for):
        body = 'При обновлении карты \'{map_is_for}\' произошла ошибка'.format(
            map_is_for=map_is_for,
        )

        self._send_message(confirm_email, body)

    def cut(self):
        src_file = fetch_file_from_storage(self.floor_map.file_name)
        stream = BytesIO(src_file)
        map_image = Image.open(stream)

        tiles = slice_map(
            map_image,
            min_zoom=self.floor_map.min_zoom,
            max_zoom=self.floor_map.max_zoom,
            zero_zoom=self.floor_map.zero_zoom,
            tile_size=self.floor_map.tile_size,
        )

        for zoom, x, y, tile in tiles:
            self.save(zoom, x, y, tile)

    def save(self, zoom, x, y, tile):
        stream = BytesIO()
        tile.save(stream, 'PNG')
        stream.seek(0)
        file_name = self.name_template.format(
            file_name=self.floor_map.file_name, x=x, y=y, z=zoom
        )
        save_file_to_storage(file_name, stream)


def populate_runner():
    sorted_floormaps = FloorMap.objects.order_by('floor', '-created_at')
    floor_id, last_floor_id = None, None

    for floor_map in sorted_floormaps:
        floor_id = floor_map.floor_id
        use_tilecutter = True if floor_id != last_floor_id else False
        save_image_to_mds(floor_map.file_name)
        if use_tilecutter:
            tile_cutter = TileCutter
            tile_cutter.delay(map_id=floor_map.id, nolock=True)
        last_floor_id = floor_id


def update_staff_office_logs(hours: int):
    today = date.today()
    now = timezone.now()

    last_logs = (
        StaffLastOffice
        .objects
        .filter(updated_at__gte=now - timedelta(hours=hours), staff__office_id=F('office_id'))
        .exclude(staff__room_id=None)
        .exclude(
            staff_id__in=(
                StaffOfficeLog
                .objects
                .filter(date=today, source=SourceTypes.WHISTLAH.value)
                .values('staff_id')
            )
        )
        .values('staff_id', 'office_id')
    )

    to_create = []
    for log in last_logs:
        to_create.append(
            StaffOfficeLog(
                staff_id=log['staff_id'],
                office_id=log['office_id'],
                date=today,
                source=SourceTypes.WHISTLAH.value,
            )
        )

    StaffOfficeLog.objects.bulk_create(to_create)


@app.task(ignore_result=True)
class UpdateStaffOfficeLogs(LockedTask):
    def locked_run(self, hours: int = 1, *args, **kwargs):
        update_staff_office_logs(hours)


def get_map_room_id_to_table_count():
    rooms = (
        Room
        .objects
        .filter(room_type=ROOM_TYPES.OFFICE, intranet_status=1)
        .values('id')
        .annotate(table_count=Count('table', filter=Q(table__intranet_status=1)))
    )
    return {room['id']: room['table_count'] for room in rooms}


def calculate_room_usage(
    room_id: int,
    staff_count: int,
    seats: int,
    room_square: float,
    source: SourceTypes,
    update_date: date
) -> Optional[RoomUsage]:
    if not seats:
        logger.warning('Not found seats in room: %s', room_id)
        return None
    usage = staff_count / seats

    required_square_per_person = settings.REQUIRED_SQUARE_PER_PERSON
    if room_square:
        square_based_usage = staff_count * required_square_per_person / room_square
    else:
        logger.warning('Room square is 0 in room: %s', room_id)
        return None

    return RoomUsage(
        room_id=room_id,
        usage=usage,
        square_based_usage=square_based_usage,
        source=source,
        date=update_date,
    )


def update_rooms_usage(source=None, update_date=None):
    update_date = update_date or date.today() - timedelta(days=1)
    source = source or SourceTypes.PACS.value

    map_room_id_to_count_table = get_map_room_id_to_table_count()
    map_room_id_to_room_square = get_map_room_id_to_room_square()

    logs = (
        StaffOfficeLog
        .objects
        .values('staff__room_id')
        .filter(source=source, date=update_date, is_weekend=False)
        .exclude(staff__room_id=None)
        .annotate(staff_count=Count('staff'))
    )

    to_create = []
    for log in logs:
        usage = calculate_room_usage(
            room_id=log['staff__room_id'],
            staff_count=log['staff_count'],
            seats=map_room_id_to_count_table.get(log['staff__room_id']),
            room_square=map_room_id_to_room_square.get(log['staff__room_id']),
            source=source,
            update_date=update_date,
        )
        if usage is not None:
            to_create.append(usage)

    RoomUsage.objects.bulk_create(to_create)


@app.task(ignore_result=True)
class UpdateRoomUsage(LockedTask):
    def locked_run(self, *args, **kwargs):
        update_rooms_usage()


def update_staff_office_logs_pacs(attachments_process_qty: int = 1):
    issue = settings.PACS_STARTREK_ISSUE
    session = requests.Session()
    session.headers['Authorization'] = f'OAuth {settings.ROBOT_STAFF_OAUTH_TOKEN}'
    try:
        response = session.get(
            url=f'{settings.STARTREK_API_URL}/v2/issues/{issue}/attachments/',
            timeout=(0.5, 2, 5),
        )
        response.raise_for_status()
    except requests.RequestException:
        logger.error('update_staff_office_logs_pacs: error loading attachments')
        return

    result = response.json()
    attachments = result[-attachments_process_qty:]
    for attachment in attachments:
        content_url = attachment['content']
        try:
            response = session.get(
                url=content_url,
                timeout=(0.5, 2, 5),
            )
            response.raise_for_status()
        except requests.RequestException:
            logger.exception('update_staff_office_logs_pacs: error loading attachment for content_url=%s', content_url)
            continue

        try:
            load_content_to_staff_office_logs(content=response.content)
        except KeyError:
            logger.exception('update_staff_office_logs_pacs: incorrect data for content_url=%s', content_url)
        except IntegrityError:
            logger.error('load_staff_office_logs content_url=%s', content_url)


def load_content_to_staff_office_logs(content):
    result = content.decode('utf-8')
    reader = csv.DictReader(result.splitlines())
    parsed_staff_office_logs_data = [
        {
            'login': log['login'],
            'office_id': log['id'],
            'date': datetime.strptime(log['Дата'], '%Y-%m-%d').date(),
        }
        for log in reader
    ]

    load_staff_office_logs(parsed_staff_office_logs_data)


@app.task(ignore_result=True)
class UpdateStaffOfficeLogsPacs(LockedTask):
    def locked_run(self, *args, **kwargs):
        update_staff_office_logs_pacs()
