from PIL import Image, PngImagePlugin
from decimal import Decimal, ROUND_UP


# Допустимый размер метаданных для .png картинок - 100 МБ
PngImagePlugin.MAX_TEXT_CHUNK = 100 * 1024 * 1024


def round_up(value):
    return Decimal(value).quantize(1, ROUND_UP)


def picture_is_white(picture):
    histogram = picture.histogram()
    return all(histogram[i] == 65536 for i in [255, 511, 767])


class NoTileException(Exception):
    pass


class MapSlicer(object):

    def __init__(self, map_image, zoom, zero_zoom, tile_size):
        self.map_image = map_image
        self.zoom = zoom
        self.zero_zoom = zero_zoom
        self.tile_size = tile_size
        self._is_resized = False
        self._is_boxed = False

    @property
    def map_size(self):
        size = tuple(self.map_image.size)
        return size

    @property
    def max_map_side(self):
        side = max(self.map_size)
        return side

    @property
    def resize_ratio(self):
        ratio = Decimal(2) ** (self.zoom - self.zero_zoom)
        return ratio

    @property
    def zoomed_size(self):
        size = tuple(int(s * self.resize_ratio) for s in self.map_size)
        return size

    @property
    def box_side(self):
        return 2 ** self.zoom * self.tile_size

    @property
    def paste_coordinates(self):
        box_side = self.box_side
        map_size = self.map_size
        x = (box_side - map_size[0]) // 2
        y = (box_side - map_size[1]) // 2
        return x, y

    @property
    def tile_side_count(self):
        count = self.max_map_side // self.tile_size
        return count

    @property
    def margin(self):
        zoom_count = 2 ** self.zoom
        margin = (zoom_count - self.tile_side_count) // 2
        return margin

    def resize(self):
        """Прежде чем резать картинку,
        ее надо уменьшить/увеличить до нужного размера.
        """
        if self.zoom == self.zero_zoom or self._is_resized:
            return
        self.map_image = self.map_image.resize(
            self.zoomed_size,
            resample=Image.ANTIALIAS,
        )
        self.is_resized = True
        return self

    def paste_in_box(self):
        """Если данная нам карта 320px а нам надо отдать 4 квадрата по 256px,
        то нам надо вписать наши 320px в центр квадарата 512px
        и заполнить пустоты белым.
        """
        box_side = self.box_side
        new_size = (box_side, box_side)

        if new_size == self.map_size or self._is_boxed:
            return

        image = Image.new(
            mode=self.map_image.mode,
            size=new_size,
            color='#ffffff',
        )
        image.paste(self.map_image, box=self.paste_coordinates)

        self.map_image = image
        self._is_boxed = True
        return self

    def _get_tile(self, x, y):
        x0 = x * self.tile_size
        y0 = y * self.tile_size
        x1 = x0 + self.tile_size
        y1 = y0 + self.tile_size

        tile = self.map_image.crop([x0, y0, x1, y1])
        if picture_is_white(tile):
            raise NoTileException

        return tile

    def get_tiles(self):
        self.resize()
        self.paste_in_box()

        count = self.tile_side_count
        margin = self.margin

        for x in range(count):
            for y in range(count):
                try:
                    yield x + margin, y + margin, self._get_tile(x, y)
                except Exception:
                    continue


def slice_map(map_image, min_zoom, max_zoom, zero_zoom, tile_size):
    for zoom in range(min_zoom, max_zoom + 1):
        slicer = MapSlicer(map_image, zoom, zero_zoom, tile_size)
        for x, y, tile in slicer.get_tiles():
            yield zoom, x, y, tile
