from itertools import chain
from typing import List, Dict, Iterable

from django.db.models import QuerySet
from django.http import JsonResponse

from staff.lib.decorators import responding_json

from staff.groups.models import Group
from staff.groups.service_update_tasks import ServiceUpdateTasks
from staff.departments.models import Department


@responding_json
def check_groups_consistency(request):
    errors = {}

    groups = Group.objects.all()

    for group in groups:
        expected_child_groups = (group.rght - group.lft - 1) // 2 + 1

        if expected_child_groups < 0:
            errors[group.id] = {
                'id': group.id,
                'url': group.url,
                'expected_count': expected_child_groups,
            }
            continue

        child_found = group.get_descendants(include_self=True).count()
        if expected_child_groups != child_found:
            errors[group.id] = {
                'id': group.id,
                'url': group.url,
                'expected_count': expected_child_groups,
                'real_count': child_found,
            }

    return errors


def _mptt_trees_problems_on_entity(entity_name: str, query_set: QuerySet) -> Iterable[str]:
    values_list = query_set.values_list(
        'url',
        'level',
        'tree_id',
        'lft',
        'rght',
        'parent_id',
        'parent__url',
        'parent__level',
        'parent__tree_id',
        'parent__lft',
        'parent__rght',
    )

    for (
        url,
        level,
        tree_id,
        left,
        right,
        parent_id,
        parent_url,
        parent_level,
        parent_tree_id,
        parent_left,
        parent_right,
    ) in values_list:
        if parent_id is None:
            continue

        if not (
            level == parent_level + 1
            and parent_left < left < right < parent_right
            and tree_id == parent_tree_id
        ):
            yield (
                f'{entity_name} {url}, lvl {level}, tree_id {tree_id}, left {left}, right {right}. ',
                f'Parent {parent_url}, lvl {parent_level}, tree_id {parent_tree_id}, lvl {parent_level}, ',
                f'left {parent_left}, right {parent_right}',
            )


def mptt_trees_problems() -> List:
    problems = list(
        chain(
            _mptt_trees_problems_on_entity('Department', Department.objects),
            _mptt_trees_problems_on_entity('Department', Department.valuestreams),
            _mptt_trees_problems_on_entity('Department', Department.geography),
            _mptt_trees_problems_on_entity('Group', Group.objects),
        ),
    )

    return problems


def check_mptt_trees(request):
    problems = mptt_trees_problems()
    if problems:
        return JsonResponse(data={'problems': problems})

    return JsonResponse(data={})


def dead_abc_service_update_tasks() -> Dict:
    failed = list(ServiceUpdateTasks.get_dead_tasks().order_by('id').values_list('id', flat=True))

    if failed:
        return {'failed_tasks': failed}

    return {}
