from datetime import datetime, timedelta

from django.conf import settings
from django.http import JsonResponse
from operator import itemgetter
from itertools import groupby

from typing import Dict, List

from staff.budget_position.models import BudgetPositionAssignment
from staff.departments.models import Department
from staff.lib.decorators import responding_json
from staff.lib import waffle
from staff.monitorings.models import FailedBpCodesUpdates
from staff.oebs.models import (
    Office,
    Employee,
    LeaveBalance,
    Organization,
    BusinessCenter,
    HeadcountPosition,
    OebsHeadcountPosition,
)
from staff.person.models import Staff


def get_unlinked_leave_balance() -> Dict:
    unlinked_leave_balance = list(
        LeaveBalance.objects
        .filter(dis_staff__isnull=True)
        .values_list('person_guid', flat=True)
    )

    if not unlinked_leave_balance:
        return {}

    return {'unlinked_leave_balance': unlinked_leave_balance}


@responding_json
def check_unlinked_leave_balance(request):
    return get_unlinked_leave_balance()


def rollups_with_problems() -> Dict:
    model_thresholds = {
         Employee: timedelta(hours=50),
         LeaveBalance: timedelta(hours=36),
         Office: timedelta(hours=36),
         Organization: timedelta(hours=36),
         HeadcountPosition: timedelta(hours=4),
         OebsHeadcountPosition: timedelta(hours=4),
         BusinessCenter: timedelta(hours=36),
    }
    problems = {}

    for model, threshold in model_thresholds.items():
        name = model.__name__
        switch = 'rollup_oebs_{}'.format(name.lower())
        if not waffle.switch_is_active(switch):
            continue

        last_rollup = model.objects.filter(last_rollup__isnull=False).values_list('last_rollup', flat=True).first()

        if not last_rollup:
            problems[name] = 'has not rolled up yet'
        else:
            if last_rollup < datetime.now() - threshold:
                problems[name] = last_rollup and last_rollup.isoformat()

    return problems


def check_success_rollup(request):
    return JsonResponse(data=rollups_with_problems())


def get_deleted_entities_with_logins(entity_type):
    entities = (
        Staff.objects
        .filter(**{'is_dismissed': False, f'{entity_type}__intranet_status': 0})
        .values('login', f'{entity_type}_id')
        .order_by(f'{entity_type}_id')
    )

    entities_with_logins = {}
    for id_, records in groupby(entities, key=itemgetter(f'{entity_type}_id')):
        entities_with_logins[id_] = [record['login'] for record in records]

    return entities_with_logins


def get_inconsistent_entities(oebs_model):
    field = oebs_model.link_field_name
    entities = (
        oebs_model.objects
        .filter(**{
            'staff_usage__in': ['N', 'НЕТ'],
            f'{field}__staff__is_dismissed': False
        })
        .values(f'{field}__staff__login', f'{field}_id')
        .order_by(f'{field}_id')
    )

    entities_with_logins = {}
    for id_, records in groupby(entities, key=itemgetter(f'{field}_id')):
        entities_with_logins[id_] = [record[f'{field}__staff__login'] for record in records]

    return entities_with_logins


def office_and_organization_inconsistency() -> Dict:
    result = {}

    deleted_offices = get_deleted_entities_with_logins('office')
    if deleted_offices:
        result['deleted_offices'] = deleted_offices

    deleted_organizations = get_deleted_entities_with_logins('organization')
    if deleted_organizations:
        result['deleted_organizations'] = deleted_organizations

    oebs_deleted_offices = get_inconsistent_entities(BusinessCenter)
    if oebs_deleted_offices:
        result['oebs_deleted_offices'] = oebs_deleted_offices

    oebs_deleted_organizations = get_inconsistent_entities(Organization)
    if oebs_deleted_organizations:
        result['oebs_deleted_organizations'] = oebs_deleted_organizations

    return result


def check_office_and_organization_consistency(request):
    return JsonResponse(data=office_and_organization_inconsistency())


def failed_bp_code_updates() -> Dict:
    fields = ('proposal_id', 'login', 'code')
    fails = list(FailedBpCodesUpdates.objects.values(*fields))
    if fails:
        return {'fails': fails}

    return {}


def check_failed_bp_code_updates(request):
    return JsonResponse(data=failed_bp_code_updates())


def check_main_assignment_existence() -> Dict[str, List[str]]:
    active_qs = BudgetPositionAssignment.objects.active().filter(person__isnull=False)
    dismissed = list(
        active_qs
        .filter(main_assignment=True, person__is_dismissed=True)
        .values_list('person__login', flat=True)
    )

    hr_controlled_department_roots = [settings.OUTSTAFF_DEPARTMENT_ID, settings.YANDEX_DEPARTMENT_ID]
    tree_ids = Department.objects.filter(id__in=hr_controlled_department_roots).values_list('tree_id', flat=True)
    main_assignments = active_qs.filter(main_assignment=True).values_list('person_id', flat=True)
    without_main_assignment = list(
        Staff.objects
        .filter(is_dismissed=False, is_robot=False, department__tree_id__in=tree_ids)
        .exclude(pk__in=main_assignments)
        .values_list('login', flat=True)
    )

    result = {}

    if dismissed:
        result['dismissed'] = dismissed

    if without_main_assignment:
        result['without_main_assignment'] = without_main_assignment

    return result
