import logging
from bson import ObjectId
from datetime import datetime, timedelta
from itertools import groupby
from typing import Dict, List

from django.db.models import Q, QuerySet
from django.http import JsonResponse

from staff.budget_position.const import WORKFLOW_STATUS
from staff.budget_position.models import ChangeRegistry
from staff.budget_position.workflow_service.entities.workflows.proposal_workflows import (
    MoveWithoutBudgetPositionWorkflow,
)
from staff.departments.edit.constants import HR_CONTROLLED_DEPARTMENT_ROOTS
from staff.departments.tasks import PushProposalsToOEBS
from staff.departments.controllers.proposal import time_now, ProposalCtl
from staff.departments.models import ProposalMetadata
from staff.departments.edit.proposal_mongo import MONGO_COLLECTION_NAME, to_mongo_id
from staff.lib.mongodb import mongo
from staff.lib.utils.date import parse_datetime
from staff.person.models import Staff
from staff.proposal.controllers import ProposalTasks


logger = logging.getLogger(__name__)


def _get_persons(proposal_data):
    actions = proposal_data.get('persons', {}).get('actions', [])
    logins = [a['login'] for a in actions]
    return logins


def _get_tickets(proposal_data):
    tickets = proposal_data.get('tickets', {})
    return {
        'persons': tickets.get('persons', {}),
        'department': tickets.get('department_ticket') or tickets.get('department_linked_ticket') or None,
    }


def get_proposals_without_persons_tickets() -> Dict[str, List[str]]:
    now = datetime.now()
    long_time_ago = now - timedelta(days=90)
    date_filter = {'created_at': {'$gt': long_time_ago.isoformat()}}
    collection = mongo.db[MONGO_COLLECTION_NAME]

    all_proposals = list(collection.find(date_filter))
    proposal_ids = (str(p['_id']) for p in all_proposals)
    excluded_proposal_ids = set(
        ProposalMetadata.objects
        .filter(proposal_id__in=proposal_ids)
        .filter(Q(deleted_at__isnull=False) | Q(applied_at__isnull=True))
        .values_list('proposal_id', flat=True)
    )

    suspicious = {}
    for proposal_data in all_proposals:
        if str(proposal_data['_id']) in excluded_proposal_ids:
            continue
        extra_logins = set(_get_persons(proposal_data)) - set(_get_tickets(proposal_data).get('persons', {}))
        if extra_logins:
            suspicious[str(proposal_data['_id'])] = list(extra_logins)

    return suspicious


def get_locked_proposals() -> List:
    now = datetime.utcnow()
    long_time_ago = now - timedelta(minutes=90)
    lock_time_filter = {'locked': {'$lt': long_time_ago.isoformat()}}
    collection = mongo.db[MONGO_COLLECTION_NAME]

    all_proposals = list(collection.find(lock_time_filter))
    proposal_ids = (str(p['_id']) for p in all_proposals)
    proposal_ids = list(
        ProposalMetadata.objects
        .filter(proposal_id__in=proposal_ids, deleted_at=None, applied_at=None)
        .values_list('proposal_id', flat=True)
    )

    return proposal_ids


def proposals_not_pushed_to_oebs() -> List:
    old_threshold = parse_datetime(time_now()) - timedelta(hours=1)
    old_threshold = old_threshold.isoformat()[:-3]

    unpushed_proposal_ids = [
        p.proposal_id
        for p in PushProposalsToOEBS.get_proposals_to_push()
    ]

    unpushed_proposals = (
        ProposalMetadata.objects
        .filter(proposal_id__in=unpushed_proposal_ids, applied_at__lte=old_threshold)
        .order_by('applied_at')
        .values_list('proposal_id', flat=True)
    )

    return list(unpushed_proposals)


def check_pushing_to_oebs(request):
    unpushed_proposals = proposals_not_pushed_to_oebs()
    if unpushed_proposals:
        return JsonResponse(data={'unpushed': list(unpushed_proposals)})
    return JsonResponse(data={})


def proposals_with_pending_workflows() -> List[ObjectId]:
    ids = set(
        ChangeRegistry.objects
        .exclude(workflow__proposal_id=None)
        .exclude(budget_position_id=None)
        .filter(workflow__status=WORKFLOW_STATUS.PENDING)
        .values_list('workflow__proposal__proposal_id', flat=True)
    )

    return [ObjectId(oid) for oid in set(ids)]


def get_logins_for_proposals() -> Dict[str, List[str]]:
    collection = mongo.db.get_collection(MONGO_COLLECTION_NAME)
    proposal_persons_from_mongo = collection.find(
        {
            '_id': {'$in': proposals_with_pending_workflows()},
            'updated_at': {'$lte': (datetime.utcnow() - timedelta(minutes=30)).isoformat()[:-3]},
        },
        {'persons.actions': 1},
    )

    return {
        str(doc['_id']): [a['login'] for a in doc['persons']['actions']]
        for doc in proposal_persons_from_mongo
    }


def get_logins_for_budget_positions_in_proposal() -> Dict[int, List[str]]:
    qs = (
        Staff.objects
        .values_list('budget_position_id', 'login')
        .order_by('budget_position_id')
    )
    bp_to_logins = {
        bp_id: [login for _, login in pairs]
        for bp_id, pairs in
        groupby(qs, lambda x: x[0])
    }
    return bp_to_logins


def get_registry_pending_changes() -> QuerySet:
    changes_fields = ('id', 'budget_position_id', 'staff__login', 'workflow__proposal__proposal_id')
    changes_qs = (
        ChangeRegistry.objects
        .exclude(workflow__proposal_id=None)
        .exclude(budget_position_id=None)
        .exclude(workflow__code=MoveWithoutBudgetPositionWorkflow.code)
        .filter(workflow__status=WORKFLOW_STATUS.PENDING, staff__isnull=False)
        .values_list(*changes_fields)
    )
    return changes_qs


def get_proposal_bp_changes() -> Dict[str, List]:
    logins_by_proposals = get_logins_for_proposals()
    bp_to_logins = get_logins_for_budget_positions_in_proposal()
    login_to_bp = dict(Staff.objects.values_list('login', 'budget_position_id'))

    problems = []
    proposals_to_fix = set()

    for change_id, registry_budget_position_id, changing_login, proposal_id in get_registry_pending_changes():
        proposal_logins = logins_by_proposals.get(proposal_id, None)
        if proposal_logins is None:
            logger.info('Skipping check for proposal_id %s', proposal_id)
            continue

        proposal_budget_positions = {login_to_bp.get(login) for login in proposal_logins}

        if registry_budget_position_id not in proposal_budget_positions:
            proposal_logins_with_bp = [f'{login}:{login_to_bp.get(login)}' for login in proposal_logins]
            problems.append(
                f'Change: {change_id} (BP: {registry_budget_position_id}:{changing_login})\n'
                f'Proposal: ({proposal_id}, proposal_logins: {proposal_logins_with_bp}\n'
                f'BP: {registry_budget_position_id}:{bp_to_logins.get(registry_budget_position_id)}\n'
            )
            proposals_to_fix.add(proposal_id)

    if problems:
        return {'problems': problems, 'proposals_to_fix': list(proposals_to_fix)}

    return {}


def check_persons_tickets(request):
    return JsonResponse(data=get_proposals_without_persons_tickets())


def check_bp_changes(request):
    return JsonResponse(data=get_proposal_bp_changes())


def check_locked_proposals(request):
    locked_proposals = get_locked_proposals()
    if locked_proposals:
        return JsonResponse(data={'locked_proposals': locked_proposals})

    return JsonResponse(data={})


def dead_proposal_tasks() -> Dict:
    failed_tasks = list(ProposalTasks.get_dead_tasks().values('id', 'proposal_id', 'callable'))

    if failed_tasks:
        proposal_ids = [task['proposal_id'] for task in failed_tasks]
        proposals = dict(ProposalMetadata.objects.filter(id__in=proposal_ids).values_list('id', 'proposal_id'))

        errors = {
            task['id']: {'proposal': proposals.get(task['proposal_id']), 'callable': task['callable']}
            for task in failed_tasks
        }

        return errors

    return {}


def check_dead_proposal_tasks(request):
    return JsonResponse(data=dead_proposal_tasks())


def unpushed_struct_changes() -> Dict:
    finished_proposals = (
        ProposalMetadata.objects
        .filter(pushed_to_oebs=None, applied_at__lte=datetime.now() - timedelta(minutes=10))
        .exclude(deleted_at__isnull=False)
        .values_list('proposal_id', flat=True)
    )

    proposals = list(ProposalCtl.filter(
        spec={
            '_id': {'$in': [to_mongo_id(_id) for _id in finished_proposals]},
            '$or': [
                {'root_departments': root_dep}
                for root_dep in dict(HR_CONTROLLED_DEPARTMENT_ROOTS)
            ]
        },
    ))

    if not proposals:
        return {}

    return {'proposals': [ctl.proposal_id for ctl in proposals]}


def check_unpushed_struct_changes(request):
    return JsonResponse(data=unpushed_struct_changes())
