from django.conf import settings
from django.core.urlresolvers import reverse
from django.db.models import Q
from django.http import JsonResponse

from staff.budget_position.models import Workflow
from staff.budget_position.const import WORKFLOW_STATUS
from staff.departments.models import ProposalMetadata
from staff.lib import requests, tvm2


def check_export_changes_works_impl():
    path = reverse('budget-position-api:export-changes')
    url = f'https://{settings.STAFF_HOST}{path}'

    try:
        response = requests.get(
            url,
            headers={tvm2.TVM_SERVICE_TICKET_HEADER: tvm2.get_tvm_ticket_by_deploy('staff')},
            timeout=settings.EXPORT_CHANGES_TIMEOUT,
        )

        if response.status_code != 200:
            return {'err': f'bad status code {response.status_code}'}

        data = response.json()
        if not data:
            return {'err': 'empty export changes response'}
    except Exception as e:
        return {'err': str(e)}

    return {}


def check_export_changes_works(request):
    result = check_export_changes_works_impl()
    return JsonResponse(data=result)


def check_workflow_consistency(request):
    fix = bool(request.GET.get('fix-it-please', False))
    return JsonResponse(data=check_workflow_status(fix))


def check_workflow_status(fix=False):
    dt = '2019-01-10T11:11:11'
    all_proposals = ProposalMetadata.objects.filter(Q(applied_at__gte=dt) | Q(deleted_at__gte=dt))

    pending_workflows = (
        Workflow.objects
        .filter(
            proposal_id__in=all_proposals.values_list('id', flat=True),
            status=WORKFLOW_STATUS.PENDING
        )
        .select_related('proposal')
    )

    errors = []
    for workflow in pending_workflows:
        proposal_status = workflow.proposal.proposal_id
        status_should_be = WORKFLOW_STATUS.PENDING

        if workflow.proposal.applied_at:
            proposal_status += f' applied at {workflow.proposal.applied_at}'
            confirmed_workflow = (
                Workflow.objects
                .filter(
                    proposal=workflow.proposal,
                    changeregistry__id__in=workflow.changeregistry_set.values_list('id', flat=True),
                )
                .exclude(status__in=[WORKFLOW_STATUS.CANCELLED, WORKFLOW_STATUS.PENDING])
                .first()
            )
            if confirmed_workflow is not None:
                status_should_be = WORKFLOW_STATUS.CANCELLED
            else:
                status_should_be = "may be confirmed? you should check proposal"
        elif workflow.proposal.deleted_at:
            proposal_status += f' deleted at {workflow.proposal.deleted_at}'
            status_should_be = WORKFLOW_STATUS.CANCELLED
        else:
            proposal_status += ' whaaat?'

        err = f'{workflow.id} PENDING -> {status_should_be}: {proposal_status}'
        errors.append(err)

        if fix and status_should_be in WORKFLOW_STATUS:
            workflow.status = status_should_be
            workflow.save()

    if errors:
        return {'workflow': errors}

    return {}
