from datetime import timedelta

from django.contrib import admin
from django.utils.html import format_html
from django.utils.safestring import mark_safe

from staff.oebs.stored_responses_storage import s3_client, STORED_RESPONSES_BUCKET_NAME
from staff.oebs.models import (
    Bonus,
    BusinessCenter,
    Currency,
    Employee,
    Geography,
    HRProduct,
    Job,
    LeaveBalance,
    Occupation,
    Office,
    Organization,
    PaySys,
    Review,
    Reward,
    StoredResponse,
)


class OEBSAdminBase(admin.ModelAdmin):
    list_display = [
        '__str__',
        'link',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
    ]


class OEBSOrganizationModelAdmin(OEBSAdminBase):
    search_fields = (
        'org_id',
        'name_ru',
        'name_en',
    )

    list_filter = ('staff_usage',)

    def link(self, obj):
        return obj.dis_organization


class OEBSOfficeModelAdmin(admin.ModelAdmin):
    list_display = [
        '__str__',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
    ]

    search_fields = (
        'location_id',
        'location_code',
        'location_addr',
    )


class OEBSEmployeeModelAdmin(OEBSAdminBase):
    search_fields = (
        'person_guid',
        'last_name',
        'first_name',
        'middle_names',
        'dis_staff__login',
    )

    def link(self, obj):
        return obj.dis_staff


class OEBSLeaveBalanceModelAdmin(OEBSAdminBase):
    search_fields = (
        'person_guid',
        'dis_staff__last_name',
        'dis_staff__first_name',
        'dis_staff__login',
    )

    def link(self, obj):
        return obj.dis_staff


class OEBSOccupationModelAdmin(admin.ModelAdmin):
    list_display = [
        '__str__',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
    ]

    search_fields = (
        'scale_code',
        'scale_name',
        'scale_description_en',
    )


class OEBSJobModelAdmin(admin.ModelAdmin):
    list_display = [
        '__str__',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
        'is_deleted_from_oebs',
    ]

    search_fields = (
        'code',
        'name_en',
    )


class OEBSGeographyModelAdmin(admin.ModelAdmin):
    list_display = [
        '__str__',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
    ]

    search_fields = (
        'code',
        'description',
    )


class OEBSBonusModelAdmin(admin.ModelAdmin):
    list_display = [
        '__str__',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
    ]

    search_fields = (
        'name',
        'description',
    )

    fields = (
        'scheme_id',
        'schemes_line_id',
        'name',
        'description',
        'start_date',
        'end_date',
        'scheme_data',
        'category',
        'value_type',
        'value_source',
        'value',
    )

    readonly_fields = fields


class OEBSRewardModelAdmin(admin.ModelAdmin):
    list_display = [
        '__str__',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
    ]

    search_fields = (
        'name',
        'description',
    )


class OEBSReviewModelAdmin(admin.ModelAdmin):
    list_display = [
        '__str__',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
    ]

    search_fields = (
        'name',
        'description',
    )


class OEBSPaySysModelAdmin(admin.ModelAdmin):
    list_display = [
        '__str__',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
    ]

    search_fields = (
        'product_name',
        'product_id',
        'service_name',
        'service_code',
    )


class OEBSHRProductModelAdmin(admin.ModelAdmin):
    list_display = [
        '__str__',
        'last_sync',
        'last_rollup',
        'last_rollup_error',
        'rollup_warnings',
    ]

    search_fields = (
        'product_id',
        'product_name',
        'service_code',
        'service_name',
    )


class OEBSBusinessCenterAdmin(OEBSAdminBase):
    search_fields = (
        'code',
        'name_ru',
        'name_en',
    )

    list_filter = ('staff_usage',)

    def link(self, obj):
        return obj.dis_office


class StoredResponseAdmin(admin.ModelAdmin):
    list_display = ('created_at', 'entity', 'code')
    ordering = ('-created_at',)
    list_filter = ('entity', 'code')
    search_fields = ('entity', 'code')

    readonly_fields = ('download_link',)

    def download_link(self, instance):
        if not instance.s3_key:
            return mark_safe('Ссылка не доступна')

        expire_timeout = timedelta(seconds=600)
        link = s3_client().generate_presigned_url(
            'get_object',
            Params={'Bucket': STORED_RESPONSES_BUCKET_NAME, 'Key': instance.s3_key},
            ExpiresIn=expire_timeout.total_seconds(),
        )
        return format_html('<a href="{}">{}</a>', link, link)

    download_link.short_description = 'Ссылка на скачивание ответа'
    download_link.allow_tags = True


class OEBSCurrency(admin.ModelAdmin):
    list_display = [
        'code',
        'last_sync',
    ]

    search_fields = (
        'code',
    )


admin.site.register(Bonus, OEBSBonusModelAdmin)
admin.site.register(BusinessCenter, OEBSBusinessCenterAdmin)
admin.site.register(Currency, OEBSCurrency)
admin.site.register(Employee, OEBSEmployeeModelAdmin)
admin.site.register(Geography, OEBSGeographyModelAdmin)
admin.site.register(HRProduct, OEBSHRProductModelAdmin)
admin.site.register(Job, OEBSJobModelAdmin)
admin.site.register(LeaveBalance, OEBSLeaveBalanceModelAdmin)
admin.site.register(Occupation, OEBSOccupationModelAdmin)
admin.site.register(Office, OEBSOfficeModelAdmin)
admin.site.register(Organization, OEBSOrganizationModelAdmin)
admin.site.register(PaySys, OEBSPaySysModelAdmin)
admin.site.register(Review, OEBSReviewModelAdmin)
admin.site.register(Reward, OEBSRewardModelAdmin)
admin.site.register(StoredResponse, StoredResponseAdmin)
