from typing import Tuple

from staff.oebs.models import (
    Bonus,
    BusinessCenter,
    Currency,
    Employee,
    Geography,
    HRProduct,
    HeadcountPosition,
    Job,
    LeaveBalance,
    Occupation,
    OebsHeadcountPosition,
    Office,
    Organization,
    PaySys,
    PersonOccupation,
    Review,
    Reward,
)
from staff.lib.sync_tools.datagenerators import DataGenerator
from staff.lib.utils.library import Library


library = Library(
    lambda dg: dg.model,
    lambda dg: dg.model.__name__,
)


class OEBSDataGenerator(DataGenerator):
    exclude_fields = ()

    def ext_data_gen(self):
        return self.external_datasource

    def get_diff_fields(self):
        return tuple(
            set(self.model._meta.get_all_field_names()) -
            set(self.get_sync_fields()) -
            set(self.get_exclude_fields())
        )

    def get_exclude_fields(self):
        return [
            'created_at',
            'modified_at',
            'last_sync',
            'last_rollup',
            'last_rollup_error',
            'rollup_warnings',
            'st_translation_id',
        ] + list(self.exclude_fields)


@library.register
class OEBSOrganizationDataGenerator(OEBSDataGenerator):
    model = Organization
    sync_fields = ('org_id',)
    exclude_fields = ('dis_organization',)


@library.register
class OEBSOfficeDataGenerator(OEBSDataGenerator):
    model = Office
    sync_fields = ('location_id',)
    exclude_fields = ('dis_placement',)


@library.register
class OEBSEmployeeDataGenerator(OEBSDataGenerator):
    model = Employee
    sync_fields = ('person_guid',)
    exclude_fields = ('dis_staff',)


@library.register
class OEBSLeaveBalanceDataGenerator(OEBSDataGenerator):
    model = LeaveBalance
    sync_fields = ('person_guid',)
    exclude_fields = ('dis_staff',)

    def build_sync_key(self, data):
        return tuple(data[f].lower() for f in self.get_sync_fields())


@library.register
class OEBSHeadcountPositionsDataGenerator(OEBSDataGenerator):
    model = HeadcountPosition
    sync_fields = ('id',)
    exclude_fields = ()


@library.register
class OebsHeadcountPositionDataGenerator(OEBSDataGenerator):
    model = OebsHeadcountPosition
    sync_fields = ('id',)
    exclude_fields = ('dis_budget_position_assignment',)


@library.register
class OccupationDataGenerator(OEBSDataGenerator):
    model = Occupation
    sync_fields = ('scale_name',)
    exclude_fields = ()


@library.register
class JobDataGenerator(OEBSDataGenerator):
    model = Job
    sync_fields = ('code',)
    exclude_fields = ()

    def __iter__(self):
        int_objects = self.get_int_objects()

        for ext_data in self.ext_data_gen():
            sync_key = self.build_sync_key(ext_data)
            int_data = int_objects.pop(sync_key, {})

            yield ext_data, int_data, sync_key

        for sync_key, int_data in int_objects.items():
            if not int_data['is_deleted_from_oebs']:
                yield {'is_deleted_from_oebs': True}, {'is_deleted_from_oebs': False}, sync_key


@library.register
class GeographyGenerator(OEBSDataGenerator):
    model = Geography
    sync_fields = ('code',)
    exclude_fields = ()


@library.register
class BonusGenerator(OEBSDataGenerator):
    model = Bonus
    sync_fields = ('scheme_id',)
    exclude_fields = ()


@library.register
class RewardGenerator(OEBSDataGenerator):
    model = Reward
    sync_fields = ('scheme_id',)
    exclude_fields = ('dis_instance',)


@library.register
class PaySysGenerator(OEBSDataGenerator):
    model = PaySys
    sync_fields = ('name',)
    exclude_fields = ()


@library.register
class ReviewGenerator(OEBSDataGenerator):
    model = Review
    sync_fields = ('scheme_id',)
    exclude_fields = ()


@library.register
class HRProductGenerator(OEBSDataGenerator):
    model = HRProduct
    sync_fields = ('product_id',)
    exclude_fields = ()


@library.register
class BusinessCenterGenerator(OEBSDataGenerator):
    model = BusinessCenter
    sync_fields = ('code',)
    exclude_fields = ('dis_office',)


@library.register
class PersonOccupationGenerator(OEBSDataGenerator):
    model = PersonOccupation
    sync_fields = ('login',)
    exclude_fields = ('dis_staff',)


@library.register
class CurrencyGenerator(OEBSDataGenerator):
    model = Currency
    sync_fields = ('code',)
    exclude_fields = ()

    def get_diff_fields(self) -> Tuple[str]:
        return ('code', )
