import logging
import json

from collections import defaultdict
from typing import Dict

from staff.oebs.constants import PERSON_POSITION_STATUS, REPLACEMENT_TYPE
from staff.oebs.controllers.datasources.json_datasource import JsonDatasource


logger = logging.getLogger(__name__)


SUMMARY_FIELDS_MAP = {
    PERSON_POSITION_STATUS.OCCUPIED: 'occupied_new_headcount',
    PERSON_POSITION_STATUS.OFFER: 'offer_new_headcount',
    PERSON_POSITION_STATUS.VACANCY_OPEN: 'vacancy_open_new_headcount',
    PERSON_POSITION_STATUS.VACANCY_PLAN: 'vacancy_plan_new_headcount',
    PERSON_POSITION_STATUS.RESERVE: 'reserve_new_headcount',
}

SUMMARY_REPLACEMENT_FIELDS_MAP = {
    PERSON_POSITION_STATUS.OCCUPIED: 'occupied_replacement_headcount',
    PERSON_POSITION_STATUS.OFFER: 'offer_replacement_headcount',
    PERSON_POSITION_STATUS.VACANCY_OPEN: 'vacancy_open_replacement_headcount',
    PERSON_POSITION_STATUS.VACANCY_PLAN: 'vacancy_plan_replacement_headcount',
}

SUMMARY_CROSS_FIELDS_MAP = {
    PERSON_POSITION_STATUS.OCCUPIED: 'occupied_replacement_cross_headcount',
    PERSON_POSITION_STATUS.OFFER: 'offer_replacement_cross_headcount',
    PERSON_POSITION_STATUS.VACANCY_PLAN: 'vacancy_plan_replacement_cross_headcount',
    PERSON_POSITION_STATUS.VACANCY_OPEN: 'vacancy_open_replacement_cross_headcount',
}


class HeadcountPositionsDatasource(JsonDatasource):

    def get_iter(self, data):
        new_cross = None  # todo: выпилить ветку new_cross==False после релиза OEBS
        for organization in data['detOrgLimits']:
            logger.debug('Processing org_id %s', organization['global_org_id'])
            if organization['global_org_id'] > 0 and 'states' in organization:

                if new_cross is None:
                    for state in organization['states']:
                        if state['positions']:
                            new_cross = 'indx' in state['positions'][0]
                            break

                if new_cross:
                    org_summary = self._extract_org_summary(organization)
                else:
                    org_summary = self._extract_org_summary_old(organization)

                first = True
                for state in organization['states']:
                    for position in state['positions']:
                        result = self._parse_position(position, state, organization)
                        if result:
                            if first:
                                result['org_summary'] = json.dumps(org_summary)
                                first = False
                            yield result

    @staticmethod
    def _position_is_new(state):
        return state['state2'] == 'NEW'

    def _extract_org_summary(self, organization):
        org_summary = defaultdict(int)
        org_summary['total_headcount'] = organization['totalHC']

        for cross_state in organization.get('crossStates', []):
            field = SUMMARY_CROSS_FIELDS_MAP.get(cross_state['state'])
            if field:
                org_summary[field] += cross_state['stateHc']

        for state in organization['states']:
            fields_map = SUMMARY_FIELDS_MAP if self._position_is_new(state) else SUMMARY_REPLACEMENT_FIELDS_MAP
            field = fields_map.get(state['state1'])
            if field:
                org_summary[field] += state['headCount']
        return org_summary

    def _extract_org_summary_old(self, organization):
        org_summary = defaultdict(int)
        org_summary['total_headcount'] = organization['totalHC']

        for state in organization['states']:
            fields_map = SUMMARY_FIELDS_MAP if self._position_is_new(state) else SUMMARY_REPLACEMENT_FIELDS_MAP
            field = fields_map.get(state['state1'])
            if field:
                org_summary[field] += state['headCount']
            cross_field = SUMMARY_CROSS_FIELDS_MAP.get(state['state1'])
            if cross_field:
                org_summary[cross_field] += state['crossHeadCount']
        return org_summary

    @classmethod
    def _parse_position(cls, position, state, organization):
        if state['state1'] not in dict(PERSON_POSITION_STATUS):
            return None

        state_position_type = '{state1}_{state2}'.format(
            state1=state['state1'],
            state2=state['state2'],
        )

        pk = '{code}_{state}_{login}'.format(
            code=position['code'],
            state=state_position_type,
            login=position['loginCurr'],
        )

        result = {
            'id': pk,

            'relevance_date': cls.get_date(organization['relevanceDate']),

            'org_id': organization['global_org_id'],
            'org_summary': '',

            'position_status': state['state1'],
            'position_category_is_new': state['state2'] == 'NEW',
            'position_code': position['code'],
            'position_name': position['name'],
            'position_product_id': position['productID'],
            'position_geo': position['geo'],
            'position_bonus_id': position['rwsBonusID'],
            'position_reward_id': position['rwsRewardID'],
            'position_review_id': position['rwsReviewID'],
            'position_current_login': position['loginCurr'],
            'position_previous_login': position['loginPrev'],
            'position_headcount': position['hc'],
            'position_main_assignment': bool(position.get('asgFlag', True)),
            'position_index': position.get('indx', 1),
            'position_prev_index': position.get('prevIndx'),
            'position_next_index': position.get('nextIndx'),
            'position_is_crossing': bool(
                position.get('prevIndx', position.get('isCrossHc', position.get('prevIsWork')))
            ),
            'position_in_total_hc': bool(position.get('isTotalHc', False)),
            'position_replacement_type': cls._extract_replacement_type(position),
        }

        return result

    @staticmethod
    def _extract_replacement_type(position):
        key = (
            bool(position.get('isTotalHc')),
            bool(position.get('prevIndx')),
            bool(position.get('nextIndx'))
        )
        replacement_type = {
            (False, False, True): REPLACEMENT_TYPE.HAS_REPLACEMENT,
            (True, True, False): REPLACEMENT_TYPE.HC_IS_BUSY,
            (False, True, True): REPLACEMENT_TYPE.HAS_REPLACEMENT_AND_HC_IS_BUSY,
        }.get(key, REPLACEMENT_TYPE.WO_REPLACEMENT)

        return replacement_type


class HeadcountPositionsStoredResponseDatasource(HeadcountPositionsDatasource):
    """
    Этот класс можно записать в поле datasource_class класса HeadcountPosition
    чтобы получить удобный способ накатывать БП из сохранненных ответов.
    Не забывайте менять id на нужный
    """

    stored_response_id = -1

    def _oebs_request(self, request_kwargs: Dict):
        from staff.oebs.models import StoredResponse
        stored_response = StoredResponse.objects.get(id=self.stored_response_id)
        return stored_response.text
