import logging
from abc import ABCMeta, abstractmethod
from datetime import datetime
from io import BytesIO
from typing import TypeVar, Dict

from django.conf import settings

from staff.lib.requests import Session, ConnectionError, Timeout
from staff.lib.tvm2 import get_tvm_ticket_by_deploy, TVM_SERVICE_TICKET_HEADER
from staff.oebs.constants import OEBS_RESOURCE_NAME
from staff.oebs.exceptions import OEBSConnectionError
from staff.oebs.stored_responses_storage import s3_client, STORED_RESPONSES_BUCKET_NAME


logger = logging.getLogger(__name__)


def _create_oebs_session():
    session = Session()
    session.headers.update({
        TVM_SERVICE_TICKET_HEADER: get_tvm_ticket_by_deploy('oebs-api'),
        'Content-type': 'application/json',
    })

    return session


class OEBSDatasource(metaclass=ABCMeta):
    def __init__(self, object_type, method, oebs_session: Session = None):
        self.object_type = object_type
        self.path = OEBS_RESOURCE_NAME[object_type]
        self.method = method.lower()
        self._data = None
        self.oebs_session = oebs_session or _create_oebs_session()

    def __iter__(self):
        if not self._data:
            self._data = self._oebs_request(request_kwargs={})
            self._data = self._deserialize_data(self._data)
        return self.get_iter(self._data)

    def get_iter(self, data):
        for obj in data:
            yield obj

    @abstractmethod
    def _deserialize_data(self, data):
        raise NotImplementedError

    def _oebs_request(self, request_kwargs: Dict):
        request = getattr(self.oebs_session, self.method, self.oebs_session.get)
        url = f'https://{settings.OEBS_HOST}/rest/{self.path}'

        try:
            response = request(url, timeout=settings.OEBS_REQUEST_TIMEOUT, **request_kwargs)
        except (ConnectionError, Timeout) as err:
            raise OEBSConnectionError(err)

        self._save_to_s3(response.content, self.object_type, response.status_code)

        if not response.status_code == 200:
            raise OEBSConnectionError(response.content)

        return response.content

    def _save_to_s3(self, content: bytes, object_type: str, status_code: int):
        s3_key = f'{object_type}.{datetime.now().isoformat()}.txt'
        self._push_to_s3_with_retries(content, s3_key, 3)

        from staff.oebs.models import StoredResponse
        StoredResponse.objects.create(
            code=status_code,
            entity=self.object_type,
            s3_key=s3_key,
        )

    @staticmethod
    def _push_to_s3_with_retries(content: bytes, s3_key: str, max_attempts: int) -> None:
        while True:
            max_attempts -= 1
            try:
                return s3_client().upload_fileobj(BytesIO(content), STORED_RESPONSES_BUCKET_NAME, s3_key)
            except Exception:
                logger.info('Failed to save OEBS response to s3 (remaining attempts %s)', max_attempts, exc_info=True)
                if max_attempts <= 0:
                    raise


OEBSDatasourceT = TypeVar('OEBSDatasourceT', bound=OEBSDatasource)
