from datetime import date
from typing import Dict, List

from staff.person.models import Staff

from staff.oebs.controllers.datasources.json_datasource import JsonDatasource


OEBS_PAGE_SIZE = 500


class PersonOccupationDatasource(JsonDatasource):
    @classmethod
    def _is_actual_record(cls, item: Dict) -> bool:
        date_from = cls.get_date(item['dateFrom'])
        date_to = cls.get_date(item['dateTo'])
        return date_from <= date.today() < date_to and item.get('gradeName')

    def _occupation_data(self, item: Dict) -> Dict or None:
        for hist_item in item['gradeHist']:
            if self._is_actual_record(hist_item):
                return {
                    'login': item['login'],
                    'occupation': hist_item['gradeName'].split('.')[0],
                }

        return None

    def _all_logins(self) -> List[str]:
        return list(
            Staff.objects
            .filter(is_dismissed=False, is_robot=False)
            .values_list('login', flat=True)
        )

    def __iter__(self):
        logins = self._all_logins()

        for i in range(0, len(logins), OEBS_PAGE_SIZE):
            next_logins_to_request = logins[i:i+OEBS_PAGE_SIZE]
            data = self._oebs_request(request_kwargs={
                'json': {'login': next_logins_to_request}
            })

            data = self._deserialize_data(data)
            for item in data['grades']:
                occupation_data = self._occupation_data(item)
                if occupation_data:
                    yield occupation_data
