import logging

from django.db.models import Q
from typing import TypeVar

from staff.departments.models import HRProduct, Geography
from staff.person.models import Staff, Organization
from staff.map.models import Office

from staff.oebs import models as oebs_models
from staff.oebs.exceptions import ObjectNotFound, OEBSConfigurationError
from staff.lib.utils.library import Library

from staff.oebs.controllers import converters


logger = logging.getLogger(__name__)
library = Library(lambda linker: linker.oebs_model)


class LinkerBase(object):
    oebs_model = None
    oebs_queryset = None
    dis_model = None
    key_field = None
    compare_by_fields = ()
    extra_lookup = {'intranet_status': 1}

    def __init__(self, oebs_queryset=None):
        self.oebs_queryset = oebs_queryset or self.oebs_queryset

    @classmethod
    def link(cls, **params):
        instance = cls(**params)
        return instance.run_linking()

    def get_lookup(self, look_for_dis_value):
        lookup = Q(**{self.compare_by_fields[1]: look_for_dis_value})
        lookup &= Q(self.get_extra_lookup())
        return lookup

    def link_instance(self, oebs_instance):
        cmp_oebs_value = getattr(oebs_instance, self.compare_by_fields[0])
        look_for_dis_value = self.to_staff(cmp_oebs_value)

        lookup = self.get_lookup(look_for_dis_value)

        dis_instance = self.get_dis_instance(lookup)
        setattr(oebs_instance, self.key_field, dis_instance)

        oebs_instance.save()

    def get_dis_instance(self, lookup):
        try:
            return self.dis_model.objects.get(Q(lookup))
        except self.dis_model.DoesNotExist as dne:
            raise ObjectNotFound(
                dne,
                '%s[%s] not found' % (self.dis_model.__name__, lookup)
            )

    def force_link(self, oebs_id, dis_id):
        oebs_instance = self.get_oebs_queryset().get(pk=oebs_id)
        lookup = {'pk': dis_id}
        dis_instance = self.get_dis_instance(lookup)

        setattr(oebs_instance, self.key_field, dis_instance)

        oebs_instance.save()

    def run_linking(self):
        queryset = self.get_oebs_queryset()
        model_name = queryset.model.__name__
        results = {'linked': 0, 'errors': 0, 'all': 0, 'name': model_name}

        for oebs_instance in queryset:
            try:
                self.link_instance(oebs_instance)
                results['linked'] += 1

            except Exception as ex:
                msg = 'Error while linking OEBS {name} [{object_id}]: {exception}'
                logger.warning(msg.format(name=model_name, object_id=oebs_instance.pk, exception=ex))
                results['errors'] += 1

            results['all'] += 1

        msg = ('Linking of {name} ended with linked={linked}, '
               'errors={errors}, all={all}, percent={percent}')
        logger.info(msg.format(
            percent=((results['linked'] * 100) // results['all']) if results['all'] else 'undefined',
            **results
        ))
        return results

    def get_oebs_queryset(self):
        if self.oebs_queryset is not None:
            queryset = self.oebs_queryset.all()  # to copy qs
        else:
            model = self.get_model()
            if model:
                queryset = model.objects.all()
            else:
                raise OEBSConfigurationError('oebs_queryset not set')
        return queryset

    def to_staff(self, left_value):
        return left_value

    def get_extra_lookup(self):
        return Q(**self.extra_lookup)

    @classmethod
    def get_model(cls):
        model = cls.oebs_model
        if model:
            return model
        else:
            raise OEBSConfigurationError('`oebs_model` not set')


LinkerBaseT = TypeVar('LinkerBaseT', bound=LinkerBase)


@library.register
class Employee2StaffLinker(LinkerBase):
    oebs_model = oebs_models.Employee
    dis_model = Staff
    compare_by_fields = ('person_guid', 'guid')
    key_field = 'dis_staff'

    def to_staff(self, person_guid):
        return converters.guid(person_guid)


@library.register
class PersonOccupation2StaffLinker(LinkerBase):
    oebs_model = oebs_models.PersonOccupation
    dis_model = Staff
    compare_by_fields = ('login', 'login')
    key_field = 'dis_staff'


@library.register
class Organization2OrganizationLinker(LinkerBase):
    oebs_model = oebs_models.Organization
    oebs_queryset = oebs_models.Organization.objects.filter(staff_usage='Y', dis_organization__isnull=True)
    dis_model = Organization
    compare_by_fields = ('name_ru', 'name')
    key_field = 'dis_organization'


@library.register
class BusinessCenter2OfficeLinker(LinkerBase):
    oebs_model = oebs_models.BusinessCenter
    oebs_queryset = oebs_models.BusinessCenter.objects.filter(staff_usage='ДА', dis_office__isnull=True)
    dis_model = Office
    compare_by_fields = ('name_ru', 'name')
    key_field = 'dis_office'


@library.register
class LeaveBalance2StaffLinker(Employee2StaffLinker):
    oebs_model = oebs_models.LeaveBalance

    def get_oebs_queryset(self):
        if self.oebs_queryset:
            queryset = self.oebs_queryset
        else:
            queryset = self.oebs_model.objects.filter(dis_staff__isnull=True)
        self._guid_to_person_id = dict(Staff.objects.filter(
            self.get_extra_lookup(),
            guid__in={it.person_guid for it in queryset},
        ).values_list('guid', 'id'))
        return queryset

    def link_instance(self, oebs_instance):
        guid = oebs_instance.person_guid
        try:
            person_id = self._guid_to_person_id[guid]
        except KeyError as exc:
            raise ObjectNotFound(exc, 'Person with guid {} not found'.format(guid))
        oebs_instance.dis_staff_id = person_id
        oebs_instance.save()


@library.register
class HRProduct2HRProductLinker(LinkerBase):
    oebs_model = oebs_models.HRProduct
    oebs_queryset = (
        oebs_models.HRProduct.objects
        .filter(dis_hr_product=None)
        .exclude(service_abc=None)  # не накатываем продукты, не привязанные к сервису (к vs).
    )
    dis_model = HRProduct
    compare_by_fields = ('product_id', 'id')
    key_field = 'dis_hr_product'


@library.register
class Geography2GeographyLinker(LinkerBase):
    oebs_model = oebs_models.Geography
    oebs_queryset = oebs_models.Geography.objects.filter(staff_geography=None)
    dis_model = Geography
    compare_by_fields = ('code', 'oebs_code')
    key_field = 'staff_geography'
    extra_lookup = {}
