from datetime import datetime
from typing import Set

from django.db import models

from staff.person.controllers import Person

from staff.lib.sync_tools.rollupper import Rollupper

from staff.oebs import models as oebs_models
from staff.oebs.controllers import converters as conv
from staff.oebs.controllers.mappers import EmployeeDataMapper


class EmployeeRollupper(Rollupper):
    model = oebs_models.Employee
    data_mapper_class = EmployeeDataMapper
    link_field_name = 'dis_staff'
    key_field_name = 'person_guid'
    rollup_rel_fields = (
        ('nda_end_date', {'extra_field_name': None, 'conv_function': conv.datetime_from_string}),
        ('contract_end_date', {'conv_function': conv.datetime_from_string}),
        ('byod_access', {'conv_function': conv.yes_no}),
        ('first_name', {'extra_field_name': 'oebs_first_name'}),
        ('middle_names', {'extra_field_name': 'oebs_middle_name'}),
        ('last_name', {'extra_field_name': 'oebs_last_name'}),
        ('concatenated_address', {'extra_field_name': 'oebs_address'}),
        ('manage_org_name', {'extra_field_name': 'oebs_manage_org_name'}),
        ('headcount', {'extra_field_name': 'oebs_headcount', 'conv_function': bool}),
        ('wiretap', {'conv_function': conv.yes_no}),
        ('staff_agreement', {'conv_function': conv.yes_no}),
        ('staff_biometric_agreement', {'conv_function': conv.yes_no}),
    )

    ROLLUP_FIELDS_FOR_FIRED = ['staff_agreement']

    def generic_rollup(
        self,
        oebs_instance,
        dis_instance,
        field_name,
        dry_run: bool,
        extra_field_name=None,
        conv_function=lambda x: x,
    ):
        if (field_name not in self.ROLLUP_FIELDS_FOR_FIRED) and self._employee_is_fired(oebs_instance):
            return False

        if extra_field_name is None:
            extra_field_name = field_name

        oebs_field_value = getattr(oebs_instance, field_name)
        value = conv_function(oebs_field_value)
        person = Person(dis_instance)
        if getattr(person, extra_field_name) == value:
            return False

        setattr(person, extra_field_name, value)
        if not dry_run:
            person.save()
        return True

    def rollup_plain_fields(
        self,
        oebs_instance: oebs_models.Employee,
        dis_instance: models.Model,
        dry_run: bool,
    ) -> Set[str]:
        if self._employee_is_fired(oebs_instance):
            return set()

        return super().rollup_plain_fields(oebs_instance, dis_instance, dry_run)

    @staticmethod
    def _employee_is_fired(oebs_instance: oebs_models.Employee) -> bool:
        return (
            oebs_instance
            and oebs_instance.actual_termination_date
            and conv.datetime_from_string(oebs_instance.actual_termination_date) < datetime.now()
        )
