import logging
from itertools import groupby

from staff.departments.models import HeadcountPosition
from staff.oebs.constants import REPLACEMENT_TYPE


logger = logging.getLogger(__name__)


def fix_zero_hc_cross():
    wrong_positions = _get_wrong_positions()
    fixed_positions = set(_fixed_zero_hc_cross_gen(wrong_positions))
    for position in fixed_positions:
        position.save()
    logger.info('fix_zero_hc_cross: %s fixed', len(fixed_positions))


def _get_wrong_positions():
    wrong_positions = (
        HeadcountPosition.objects
        .filter(code__in=(
            # Positions wo HC, but has replacement
            HeadcountPosition.objects
            .filter(headcount=0)
            .exclude(replacement_type=REPLACEMENT_TYPE.WO_REPLACEMENT)
            .values('code')
        ))
        .order_by('code', 'index')
    )
    return wrong_positions


def _fixed_zero_hc_cross_gen(wrong_positions):
    for code, positions in groupby(wrong_positions, key=lambda p: p.code):
        prev_pos = None
        for pos in positions:
            if pos.headcount == 0:
                if prev_pos:
                    _fix_active_pos_before_zero_hc(prev_pos)
                    yield prev_pos
                _fix_zero_hc_pos(pos)
                yield pos
            else:
                if prev_pos:
                    _fix_active_pos_after_active_pos(pos, prev_pos)
                    yield pos
                    if prev_pos.next_index is None:
                        _fix_active_pos_before_active_pos(pos, prev_pos)
                        yield prev_pos
                elif pos.prev_index:
                    _fix_active_pos_after_zero_hc(pos)
                    yield pos
                prev_pos = pos


def _fix_active_pos_before_zero_hc(pos):
    pos.next_index = None
    pos.replacement_type = (
        REPLACEMENT_TYPE.HC_IS_BUSY
        if pos.prev_index else
        REPLACEMENT_TYPE.WO_REPLACEMENT
    )


def _fix_zero_hc_pos(pos):
    pos.prev_index = None
    pos.index = None
    pos.next_index = None
    pos.replacement_type = REPLACEMENT_TYPE.WO_REPLACEMENT


def _fix_active_pos_after_active_pos(pos, prev_pos):
    pos.prev_index = prev_pos.index
    pos.index = pos.prev_index + 1
    pos.next_index = pos.index + 1 if pos.next_index else None


def _fix_active_pos_before_active_pos(pos, prev_pos):
    prev_pos.next_index = pos.index
    prev_pos.replacement_type = (
        REPLACEMENT_TYPE.HAS_REPLACEMENT_AND_HC_IS_BUSY
        if prev_pos.prev_index else
        REPLACEMENT_TYPE.HAS_REPLACEMENT
    )


def _fix_active_pos_after_zero_hc(pos):
    pos.prev_index = None
    pos.index = 1
    pos.next_index = pos.index + 1 if pos.next_index else None
    pos.replacement_type = (
        REPLACEMENT_TYPE.HAS_REPLACEMENT
        if pos.next_index else
        REPLACEMENT_TYPE.WO_REPLACEMENT
    )
