import random
from staff.lib import waffle
import yenv

from django.conf import settings
from django.db.models import Case, When, Value

from staff.departments.models import HeadcountPosition
from staff.person.models import Staff


def obfuscate_headcounts_data():
    if yenv.type == 'production':
        return

    if waffle.switch_is_active('disable_headcounts_mixer_in_testing'):
        return

    _obfuscate_logins()
    _obfuscate_departments()


def _all_logins_from_positions():
    result = []

    qs = (
        HeadcountPosition.objects
        .filter(current_person__isnull=False)
        .values_list('current_login', 'previous_login')
    )

    for current_login, previous_login in qs:
        result.append(current_login)
        if previous_login:
            result.append(previous_login)
    return result


def _shuffle_logins(logins):
    persons = dict(Staff.objects.filter(login__in=logins).values_list('login', 'id'))

    current_login_whens = []
    current_person_id_whens = []
    previous_login_whens = []
    previous_person_id_whens = []

    for p in HeadcountPosition.objects.filter(current_person__isnull=False):
        login = logins.pop()
        current_login_whens.append(When(id=p.id, then=Value(login)))
        current_person_id_whens.append(When(id=p.id, then=persons.get(login)))
        if p.previous_login:
            login = logins.pop()
            previous_login_whens.append(When(id=p.id, then=Value(login)))
            previous_person_id_whens.append(When(id=p.id, then=persons.get(login)))

    HeadcountPosition.objects.update(
        current_login=Case(*current_login_whens, default=None),
        current_person_id=Case(*current_person_id_whens, default=None),
        previous_login=Case(*previous_login_whens, default=None),
        previous_person_id=Case(*previous_person_id_whens, default=None),
    )


def _obfuscate_logins():
    logins = _all_logins_from_positions()
    random.shuffle(logins)
    _shuffle_logins(logins)


def _obfuscate_departments():
    departments_ids = list(set(HeadcountPosition.objects.values_list('department_id', flat=True)))
    shuffled_departments_ids = random.sample(departments_ids, len(departments_ids))

    for from_department_id, to_department_id in zip(departments_ids, shuffled_departments_ids):
        _swap_departments_ids(HeadcountPosition, from_department_id, to_department_id)


def _swap_departments_ids(cls, first_department_id, second_department_id):
    cls.objects.filter(department_id=first_department_id).update(department_id=settings.ROBOT_DEPARTMENT_ID)
    cls.objects.filter(department_id=second_department_id).update(department_id=first_department_id)
    cls.objects.filter(department_id=settings.ROBOT_DEPARTMENT_ID).update(department_id=second_department_id)
