from datetime import datetime
import logging

from staff.departments.controllers.value_streams_rollup import ValueStreamsRollupController
from staff.departments.models import HRProduct, Department
from staff.groups.models import Group, GROUP_TYPE_CHOICES

from staff.lib.sync_tools.data_mapper import DataMapper
from staff.lib.sync_tools.rollupper import Rollupper

from staff.oebs import models as oebs_models
from staff.oebs.controllers.rolluppers.update_intranet_status import map_end_date_to_intranet_status


logger = logging.getLogger(__name__)


def service_id(value):
    return int(value)


def lookup_value_stream_id(abc_id: str or None) -> int or None:
    """
    По abc-шному айдишнику сервиса ищет соответствующую сервисную группу
    Если вдруг abc_id=None или группа не найдена, то упадём
    выше по стеку из-за констрейна not null и не накатим сущность.
    """
    if not abc_id:
        return None
    abc_id = service_id(abc_id)

    abc_service = (
        Group.objects
        .filter(intranet_status=1, type=GROUP_TYPE_CHOICES.SERVICE)
        .filter(service_id=abc_id)
        .first()
    )

    if not abc_service:
        logger.info('Got non existent service id %s from OEBS in HRProduct', abc_id)
        return None

    result = (
        Department.valuestreams
        .filter(url=abc_service.url)
        .values_list('id', flat=True)
        .first()
    )

    if not result:
        logger.info('Valuestream was not found for service %s', abc_service.url)

    return result


class HRProductDataMapper(DataMapper):
    mapping = (
        ('product_id', 'id'),
        ('end_date', 'end_date'),
        ('end_date', 'intranet_status', map_end_date_to_intranet_status),
        ('product_name', 'product_name'),
        ('service_abc', 'service_id', service_id),
        ('st_translation_id', 'st_translation_id'),
        ('service_abc', 'value_stream_id', lookup_value_stream_id),
    )


class HRProductRollupper(Rollupper):
    model = oebs_models.HRProduct
    queryset = oebs_models.HRProduct.objects.exclude(service_abc=None)
    data_mapper_class = HRProductDataMapper
    link_field_name = 'dis_hr_product'
    key_field_name = 'product_id'

    create_absent = True

    def create_dis_instance(self, oebs_instance: oebs_models.HRProduct) -> HRProduct:
        now = datetime.now()
        return HRProduct(
            id=oebs_instance.product_id,
            created_at=now,
            modified_at=now,
            intranet_status=1,
            product_name=oebs_instance.product_name,
            end_date=oebs_instance.end_date,
            value_stream_id=lookup_value_stream_id(oebs_instance.service_abc),
            service_id=service_id(oebs_instance.service_abc),
        )

    def run_rollup(self, object_id=None, dry_run=False):
        super().run_rollup(object_id, dry_run)
        ValueStreamsRollupController().rollup()
