from typing import Set

from staff.lib.sync_tools.data_mapper import DataMapper
from staff.lib.sync_tools.rollupper import Rollupper
from staff.map.models import COUNTRY_CODES
from staff.person.controllers import Person
from staff.person.models import Staff

from staff.oebs import models as oebs_models


def days_off_and_vacancies(value_as_string):
    if not value_as_string:
        return 0.0
    return float(value_as_string.replace(',', '.'))


class LeaveBalanceDataMapper(DataMapper):
    mapping = (
        ('leave_balance_default', 'vacation', days_off_and_vacancies),
        ('leave_balance_company', 'extra_vacation', days_off_and_vacancies),
        ('time_off', 'paid_day_off', days_off_and_vacancies),
    )


class LeaveBalanceRollupper(Rollupper):
    model = oebs_models.LeaveBalance
    data_mapper_class = LeaveBalanceDataMapper
    link_field_name = 'dis_staff'
    key_field_name = 'person_guid'
    rollup_rel_fields = []

    def generic_rollup(
        self,
        oebs_instance,
        dis_instance,
        field_name,
        dry_run: bool,
        extra_field_name=None,
        conv_function=lambda x: x,
    ):
        if extra_field_name is None:
            extra_field_name = field_name

        oebs_field_value = getattr(oebs_instance, field_name)
        value = conv_function(oebs_field_value)
        person = Person(dis_instance)
        if getattr(person, extra_field_name) == value:
            return False

        setattr(person, extra_field_name, value)

        if not dry_run:
            person.save()

        return True

    def rollup_plain_fields(
        self,
        oebs_instance: oebs_models.LeaveBalance,
        dis_instance: Staff,
        dry_run: bool,
    ) -> Set[str]:
        # Для Беларуси данные отпусков передаются не из oebs
        country_code = dis_instance.office.get_country_code()
        if country_code.lower() == COUNTRY_CODES.BELARUS.lower():
            return set()

        return super().rollup_plain_fields(oebs_instance, dis_instance, dry_run)
