import logging

from django.db import models

from staff.oebs import models as oebs_models
from staff.map.models import Office, Placement
from staff.person.models import Organization
from staff.lib.sync_tools.rollupper import Rollupper
from staff.oebs.controllers.mappers import PlacementDataMapper


logger = logging.getLogger(__name__)


class PlacementRollupper(Rollupper):
    model = oebs_models.Office
    data_mapper_class = PlacementDataMapper
    link_field_name = 'dis_placement'
    key_field_name = 'id'
    rollup_rel_fields = (
        ('taxunit_code', {}),
        ('businesscentre_code', {}),
    )

    def create_dis_instance(self, oebs_instance: models.Model) -> models.Model:
        return Placement()

    def generic_rollup(
        self,
        oebs_instance: oebs_models.Office,
        dis_instance: Placement,
        field_name: str,
        dry_run: bool,
        **kwargs,
    ):
        if field_name == 'taxunit_code':
            return self.setup_taxunit_relation(oebs_instance, dis_instance)

        if field_name == 'businesscentre_code':
            return self.setup_office(oebs_instance, dis_instance)

        return False

    def setup_taxunit_relation(self, oebs_instance: oebs_models.Office, dis_instance: Placement) -> bool:
        if oebs_instance.taxunit_code is None:
            return False

        organization = Organization.objects.filter(oebs_organization__org_id=oebs_instance.taxunit_code).first()

        if not organization:
            logger.warning('Can\'t find organization for placement %s', oebs_instance.location_id)
            return False

        dis_instance.organization = organization
        return True

    def setup_office(self, oebs_instance: oebs_models.Office, dis_instance: Placement) -> bool:
        if not oebs_instance.businesscentre_code:
            return False

        office = Office.objects.filter(oebs_businesscenter__code=oebs_instance.businesscentre_code).first()

        if not office:
            logger.warning('Can\'t find office for placement %s', oebs_instance.location_id)
            return False

        dis_instance.office = office
        return True
